#include <iostream>
#include <string>
#include "GapPattern.h"
#include "TriangleColumns.h"
#include "Triangle.h"


// Private Constructor
Triangle::Triangle(){
	loadColumns();
}

void Triangle::loadColumns()
{

	// Copy the "String Values" from our enumeration into a Temp Array so that we can sort it.
	std::string tempArr[Tri::pyramidDepth];
	for(int i=0; i<Tri::pyramidDepth; i++){
		tempArr[i] = Tri::getColumnStr(Tri::getEnum(i));
	}

	// Reverse the order from the Temp Array so that the lowest priority is on top.
	for(int i=0; i<Triangle::MAX_TRIANGLES; i++){
		this->columnNames[Triangle::MAX_TRIANGLES - 1 - i] = tempArr[i];
	}

	// Reset the Cache Flags for the Gap Stuctures.
	for(int i=0; i < Triangle::MAX_TRIANGLES; i++){
		this->gapPatternsCache_FLAGS[i] = false;
	}
}

// The lower the value, the lower the priority for matching within the C27 Query.
// Accepts values 1-27
std::string Triangle::getColumnNameByDepth(const int depth) const
{
	this->validateDepth(depth);

	return this->columnNames[depth - 1];
}

int Triangle::getChainLengthFromDepth(const int depth) const
{
	std::string colName = this->getColumnNameByDepth(depth);
	std::string prefix = "L";

	if(prefix[0] != colName[0]){
		throw "The prefix of the column name must be an uppercase 'L'";
	}

	if(colName.size() < 2){
		throw "The column name must have at least 2 characters";
	}

	if(!isdigit(colName[1])){
		throw "The character following the prefix 'L' must be a digit.";
	}

	int retValue = static_cast<int>(colName[1]) - 48;

	return retValue;
}

// This is almost like getChainLengthFromDepth() except that it will tell you the number of Conext Items between the start and the last element (regardless of gaps)
// For example the following "CONTEXT CHAIN" has an absolute length of 6 (even though the "CONTEXT CHAIN LENGTH" is 2)	EX: "F + A"
int Triangle::getAbsoluteLengthByDepth(const int depth)
{
	this->validateDepth(depth);

	GapPattern gapsStruct = this->getGapPattern(depth);

	int counter = 0;
	int maxCounter = 0;

	for each(bool gapFlag in gapsStruct.gaps){

		counter++;

		if(!gapFlag){
			maxCounter = counter;
		}
	}

	return maxCounter;
}


// This will return an array with Boolean values.
// Every "CONTEXT DEPTH" has a "CHAIN LENGTH" value, as well as a pattern of gaps.
// A value of TRUE indicates that a "CONTEXT GAP" exists so the value will not be part of a "CONTEXT CHAIN"
// The first "CONTEXT GAP" value will always be FALSE because the "ROOT CHAIN LINK" must exist immediately before a prediction.
// This method will always return a consistent "array length".

// Here are some examples of what the "return arrays" would look like for the corresponding "CONTEXT DEPTHS".
// This assumes a conversation begins with "A".  The last thing said is "F".
// -------------------------------------------------------
// "CONTEXT DEPTH"		 	27
// "CONTEXT COLUMN NAME"	L6
// "CONTEXT CHAIN":			F + E + D + C + B + A
// array(false, false, false, false, false, false)
// -------------------------------------------------------
// "CONTEXT DEPTH"		 	22
// "CONTEXT COLUMN NAME"	L4_G01
// "CONTEXT CHAIN":			F + E + C + B
// array(false, false, true, false, false, true)
// -------------------------------------------------------
// "CONTEXT DEPTH"		 	2
// "CONTEXT COLUMN NAME"	L2_G4
// "CONTEXT CHAIN":			F + A
// array(false, true, true, true, true, false)
// -------------------------------------------------------
GapPattern& Triangle::getGapPattern(const int depth)
{
	this->validateDepth(depth);

	// Find out if the results have been cached in our static array.
	if(Triangle::gapPatternsCache_FLAGS[depth -1]){
		return Triangle::gapPatternsCache_OBJ[depth -1];
	}

	const std::string colName = this->getColumnNameByDepth(depth);

	std::string gapStr;

	if(colName.size() > 2){

		if(colName.size() < 5 || colName.size() > 7){
			throw "If a Gap is defined in the column name, the Gap Number(s) should start on the 5th character and they should not go past the 7th.";
		}

		gapStr = colName.substr(4, colName.size());
	}

	GapPattern gapsStruct;

	// The first "CONTEXT GAP" value will always be FALSE because the "ROOT CHAIN LINK" must exist immediately before a prediction.
	gapsStruct.gaps[0] = false;

	const int specifiedChainLength = this->getChainLengthFromDepth(depth);

	// We already set the first array element to true.
	// The highest that this value can go is 6, on "L6".
	int chainLinkCounter = 1;

	bool inBetweenGaps = false;

	std::size_t gapsExtracted = 0;
	int lastGapValue = 0;

	// Start counting at 1 because the first element is always FALSE.
	for(int i=1; i < GapPattern::MAX_GAP_POSITIONS; i++){

		if(chainLinkCounter >= specifiedChainLength){
			gapsStruct.gaps[i] = true;
			continue;
		}

		// If there are 2 gaps within a column name (such as L3_G21), we need to allow one "CHAIN LINK" to be filled between the "2" and the "1".
		if(!inBetweenGaps){

			// Check and see if there is another Gap value to extract.
			if(0 == lastGapValue){

				// Don't allow an out of range exception.
				if(gapStr.size() > gapsExtracted){

					std::string lastGapDigitStr = gapStr.substr(gapsExtracted, 1);

					// Convert the ASCII character code into an <int>.
					lastGapValue = static_cast<int>(lastGapDigitStr[0]) - 48;

					if(lastGapValue < 0 || lastGapValue > 9){
						throw "The Gap Value is out of range.";
					}
				}
				else{
					lastGapValue = 0;
				}

				gapsExtracted++;
			}

			// Until we finish cycling over all of the gaps, we need to fill the array with TRUE entries to indicate empty spots.
			if(0 != lastGapValue){

				gapsStruct.gaps[i] = true;
				lastGapValue--;

				if(0 == lastGapValue){
					inBetweenGaps = true;
				}

				continue;
			}
		}

		// This means that there isn't a gap at the current position.
		// Add a FALSE entry to the array and increment the size of our ChainLink counter.
		gapsStruct.gaps[i] = false;
		chainLinkCounter++;
		inBetweenGaps = false;
	}

	if(0 != lastGapValue){
		throw "The Gap Value is incorrect at given depth";
	}

	if(chainLinkCounter != this->getChainLengthFromDepth(depth)){
		throw "The number of chain links detected by the Gap sequence did not match the corresponding L value at given depth";
	}

	// To increase performance a bit, add the results to a static arrary in case this method is called again with the same depth.
	this->gapPatternsCache_OBJ[depth -1] = gapsStruct;
	this->gapPatternsCache_FLAGS[depth -1] = true;

	return this->gapPatternsCache_OBJ[depth -1];
}


// Private Method
void Triangle::validateDepth(const int depth) const
{
	if(depth > Triangle::MAX_TRIANGLES || depth < Triangle::START_TRIANGLE){
		throw "Illegal Context Depth";
	}
}
