#include <iostream>
#include <string>
#include <fstream>
#include <vector>
#include "SignatureStruct.h"
#include "TriangleColumns.h"
#include "QueryStruct.h"
#include "Cortex.h"
#include "CortexSearch.h"



// STATIC Method
// Sorts the characters in a string alpha-numerically by reference.
void CortexSearch::sortStr(std::string& strToSrt){

	std::string tempChar1;
	std::string tempChar2;

	bool swapped = false;

	while(true){

		for(std::size_t i=0; i < (strToSrt.size() - 1); i++){

			// Find out if the ASCII code is greater (which are integers ordered alphanumerically
			if(std::size_t(strToSrt[i]) > std::size_t(strToSrt[i+1])){

				swapped = true;

				tempChar1 = strToSrt.substr(i, 1);
				tempChar2 = strToSrt.substr((i+1), 1);

				strToSrt[i+1] = tempChar1[0];
				strToSrt[i] = tempChar2[0];
			}
		}

		if(!swapped){
			break;
		}

		swapped = false;
	}
}

// Returns an index position (starting at Zero) for where the given signature should be inserted.
// If there is already a matching "needle", then this method will return -1.
// The insertion looks for the record which is 1 higher.  If it finds one, it inserts before (which could be at Array[0].
// If it can't find one that is 1 higher, then it will insert at the very end of the array.  In which case, the "Index Position" returned by this method would be equal to the size of the array elements.
long CortexSearch::getIndexPositionForInsertion(std::fstream& fileHandle, const std::streamsize startingByteOffset, const SignatureStruct& needle, const long maxRecords)
{
	if(!fileHandle.is_open()){
		throw "The file handle is not open when trying to get index position.";
	}

	// If there are no records yet, then insertion should happen at the first record (zero-based).
	if(maxRecords <= 0){
		return 0;
	}

	fileHandle.seekg(startingByteOffset, std::ios::beg);

	long left = 0;
	long right = maxRecords;


	Cortex::Index cortexIndex={0};

	while(left <= right && left < maxRecords){

		long middle = (left + right) / 2;

		fileHandle.seekp(startingByteOffset + sizeof(Cortex::Index) * middle, std::ios::beg);
		fileHandle.read((char*)&cortexIndex, sizeof(Cortex::Index));

		if(!fileHandle){
			throw "Error in getIndexPositionForInsertion while advancing the read pointer.";
		}

		if(std::strcmp(cortexIndex.recordName, needle.digestedStr.c_str()) == 0){
			// Return Negative 1 because it found a match.
			// We are not supposed to find a match if we are going to insert.
			return -1;
		}
		else if (std::strcmp(cortexIndex.recordName, needle.digestedStr.c_str()) > 0){
			right = middle - 1;
		}
		else{
			left = middle + 1;
		}
	}

	if(left < 0){
		return 0;
	}
	else{
		return left;
	}
}




// Returns the index position matching the record name (which is like an auto-increment) starting at 0.
// If the record can not be found, this method will return -1.
// This is the heart of the "binary search" for all records in the DB.  It requires all Cortex::Index structures to be stored contiguously.
long CortexSearch::getIndexPositionFromRecordName(std::fstream& fileHandle, const std::streamsize startingByteOffset, const SignatureStruct& needle, const long maxRecords)
{
	if(!fileHandle.is_open()){
		throw "The file handle is not open when trying to get index position.";
	}

	// Save a little work.
	if(maxRecords <= 0){
		return -1;
	}

	fileHandle.seekg(startingByteOffset, std::ios::beg);

	long left = 0;
	long right = maxRecords;

	Cortex::Index cortexIndex={0};

	while(left <= right){

		long middle = (left + right) / 2;

		fileHandle.seekp(startingByteOffset + sizeof(Cortex::Index) * middle, std::ios::beg);
		fileHandle.read((char*)&cortexIndex, sizeof(Cortex::Index));

		if(!fileHandle){
			throw "Error in getIndexPositionFromRecordName while advancing the read pointer.";
		}

		if(std::strcmp(cortexIndex.recordName, needle.digestedStr.c_str()) == 0){
			return middle;
		}
		else if (std::strcmp(cortexIndex.recordName, needle.digestedStr.c_str()) > 0){
			right = middle - 1;
		}
		else{
			left = middle + 1;
		}
	}

	return -1;
}

// This can be used for both insertion and fetching.  You have to compare the value of the returned "index location" to know whether the signatureNeedle exists or not.
// Sometimes you can have multiple signatures in the same C27 Column... for example some people have the same Last Name.  The LX Signature is always unique.
// If there are multiple records with a matching Signature, this will look for the very first one in the queue.
long CortexSearch::getIndexPositionForVector(std::vector<Cortex::Index>& indexVector, const SignatureStruct& signatureNeedle)
{
	long left = 0;
	long right = indexVector.size();

	// Don't access empty vector.
	if(right == 0){
		return 0;
	}

	while(left <= right){

		std::size_t middle = (left + right) / 2;

		if(strcmp(indexVector.at(middle).recordName, signatureNeedle.digestedStr.c_str()) == 0){
			return middle;
		}
		if(strcmp(indexVector.at(middle).recordName, signatureNeedle.digestedStr.c_str()) > 0){
			right = middle - 1;
		}
		else{
			left = middle + 1;
		}
	}

	long returnVal = 0;

	if(left > 0){
		returnVal =  left;
	}

	while(returnVal >= 0){

		if(strcmp(indexVector.at(returnVal).recordName, signatureNeedle.digestedStr.c_str()) == 0){

		}
	}

}


// Will return the "Byte Offset" for a Character Code Stucture.
// The value is relative to where the "Character Indexes" end.
std::streamsize CortexSearch::getByteOffsetFromCharacterIndex(std::fstream& fileHandle, std::size_t charCodeNeedle, long totalCharacters)
{
	std::streamsize byteOffset = sizeof(Cortex::Header);

	// The signature for the "Character Collection" is just a single character (versus a digestive hash).
	SignatureStruct characterSignature;
	characterSignature.digestedStr = Cortex::Globals::getCharCodeString(charCodeNeedle);

	long indexPosition = CortexSearch::getIndexPositionFromRecordName(fileHandle, byteOffset, characterSignature, totalCharacters);

	if(indexPosition < 0){
		throw "The character code could not be located within the index structure in method getByteOffsetFromCharacterIndex.";
	}

	// The Index Structure will contain the "Byte Offset" associated with the Character Stucture.
	Cortex::Index cortexIndex={0};
	Cortex::Read::fromBinFile(fileHandle, cortexIndex, (byteOffset + (sizeof Cortex::Index) * indexPosition));

	return cortexIndex.byteOffset;
}

// Will return the "Byte Offset" for a ChildSignature Stucture.
// The value is relative to where the "ChildSignature Indexes" end.
std::streamsize CortexSearch::getByteOffsetFromChildSignatureIndex(std::fstream& fileHandle, std::size_t charCodeNeedle, long totalCharacters, const SignatureStruct& childSigNeedle, long totalChildSignatures)
{

	std::streamsize byteOffset = sizeof(Cortex::Header);

	byteOffset += totalCharacters * sizeof(Cortex::Index);

	byteOffset += CortexSearch::getByteOffsetFromCharacterIndex(fileHandle, charCodeNeedle, totalCharacters);

	byteOffset += sizeof(Cortex::Character);

	// The "Byte Offset" is now set immediately underneath where the "Character Structure" ends... and it should be for the "Character Stucture" which contains our "Child Signature".
	long indexPosition = CortexSearch::getIndexPositionFromRecordName(fileHandle, byteOffset, childSigNeedle, totalChildSignatures);

	if(indexPosition < 0){
		throw "The Child Signature code could not be located within the index structure in method getByteOffsetFromChildSignatureIndex";
	}

	// The Index Structure will contain the "Byte Offset" associated with the Character Stucture that we are looking for.
	Cortex::Index cortexIndex={0};
	Cortex::Read::fromBinFile(fileHandle, cortexIndex, (byteOffset + (sizeof Cortex::Index) * indexPosition));

	return cortexIndex.byteOffset;
}


std::streamsize CortexSearch::getIndexesVector(std::fstream& fileHandle, std::vector<Cortex::Index>& indexVector, std::streamsize startingByteOffset, const long maxRecords)
{
	if(!fileHandle.is_open()){
		throw "The file handle is not open while calling method getIndexesVector.";
	}

	indexVector.clear();
	indexVector.reserve(maxRecords);

	Cortex::Index cortexIndex={0};

	// For each character code, we expect there to be an Index.
	// This will put all of the "Index Structs" into a vector.
	for(long i=0; i<maxRecords; i++){

		Cortex::Index cortexIndex={0};
		Cortex::Read::fromBinFile(fileHandle, cortexIndex, (startingByteOffset + (sizeof Cortex::Index) * i));

		// This will make a copy of the Index Structure... so don't worry about it going out of scope.
		indexVector.push_back(cortexIndex);
	}

	// Do multiplication to save the processor from adding a bunch of stuff in a loop.
	startingByteOffset += maxRecords * (sizeof Cortex::Index);

	return startingByteOffset;
}


Cortex::Header CortexSearch::getHeaderStructure(std::fstream& masterBinFileHandle)
{
	Cortex::Header cortexHeader={0};
	Cortex::Read::fromBinFile(masterBinFileHandle, cortexHeader, 0);

	if(cortexHeader.headerDesc != Cortex::Globals::CORTEX_HEADER_DESC){
		throw "This file does not appear to be a 'Cortex File'.";
	}

	return cortexHeader;
}


void CortexSearch::getCharacterStucturesFromConsolidatedFile(std::fstream& fileHandle, std::vector<Cortex::Character>& characterVector)
{
	if(!fileHandle.is_open()){
		throw "The file handle is not open while calling method getRowsVector.";
	}

	Cortex::Header cortexHeader = CortexSearch::getHeaderStructure(fileHandle);

	long maxRecords = cortexHeader.characterCount;
	characterVector.clear();
	characterVector.reserve(maxRecords);

	std::streamsize byteOffset = sizeof(Cortex::Header);
	byteOffset += sizeof(Cortex::Index) * maxRecords;

	for(long i=0; i<maxRecords; i++){

		Cortex::Character cortexChar={0};
		Cortex::Read::fromBinFile(fileHandle, cortexChar, byteOffset);

		// This will make a copy of the Index Structure... so don't worry about it going out of scope.
		characterVector.push_back(cortexChar);

		// The Character Structures are not stored contiguously.
		byteOffset += cortexChar.byteTalley;
	}
}


// This will extract the child signatures by using the "Byte Talley" value on each of the structures.
// Just make sure to position the "Byte Offset" directly where the first one starts and provide a Max Count.
// If for some reason RAM was a constraint, this could be done in chunks.
// The return value will increment the "Starting Byte Offset"... NOT just by the size of the Raw ChildSignature Structures.... but by the Bytes which all of the structure contains.
std::streamsize CortexSearch::getChildSigStuctures(std::fstream& fileHandle, std::vector<Cortex::ChildSignature>& childSigVector, std::streamsize startingByteOffset, const long totalRecordsForExtraction)
{
	if(!fileHandle.is_open()){
		throw "The file handle is not open while calling method getChildSigStuctures.";
	}

	childSigVector.clear();
	childSigVector.reserve(totalRecordsForExtraction);

	for(long i=0; i<totalRecordsForExtraction; i++){

		Cortex::ChildSignature cortexChildSig={0};
		Cortex::Read::fromBinFile(fileHandle, cortexChildSig, startingByteOffset);

		// This will make a copy of the Index Structure... so don't worry about it going out of scope.
		childSigVector.push_back(cortexChildSig);

		// The Child Signature Structures are not stored contiguously.
		startingByteOffset += cortexChildSig.byteTalley;
	}

	return startingByteOffset;
}


// Vector is passed by refference to avoid using the "new" operator.
std::streamsize CortexSearch::getRowsVector(std::fstream& fileHandle, std::vector<Cortex::Row>& rowVector, std::streamsize startingByteOffset, const long totalRecordsForExtraction){

	if(!fileHandle.is_open()){
		throw "The file handle is not open while calling method getRowsVector.";
	}

	rowVector.clear();
	rowVector.reserve(totalRecordsForExtraction);

	for(long i=0; i<totalRecordsForExtraction; i++){

		Cortex::Row cortexRow={0};
		Cortex::Read::fromBinFile(fileHandle, cortexRow, startingByteOffset + (sizeof Cortex::Row) * i);

		// This will make a copy of the Index Structure... so don't worry about it going out of scope.
		rowVector.push_back(cortexRow);
	}

	// Do multiplication to save the processor from adding a bunch of stuff in a loop (although not noticeable).
	startingByteOffset += totalRecordsForExtraction * (sizeof Cortex::Row);

	return startingByteOffset;
}

std::streamsize CortexSearch::getFileSize(std::fstream& fileHandle)
{
	if(!fileHandle.is_open()){
		throw "The file handle is not open while calling method CortexSearch::getFileSize.";
	}

	fileHandle.seekg(0, std::ios::end);

	return fileHandle.tellg();
}


std::size_t CortexSearch::getCharacterCount(std::fstream& masterBinFileHandle)
{
	Cortex::Header cortexHeader = CortexSearch::getHeaderStructure(masterBinFileHandle);

	return cortexHeader.characterCount;
}