#pragma once


// This is the reason why we sleep (dreaming is a side effect).  We write to "Cache Files" during the day and and night they are consolidated back into the master.
// Dreaming is interesting.  You are able to surf your "memory wave" on newly created indexes... however your dreams are stored purely in RAM.  That is why you can only remember up to 1-2 minutes worth.
// Anyway, don't implement these classes directly.  Use the main interface through the class "CortexCache".

// ------ BASE Compartment -------------
// This is the Base Class (which is purely virtual) for both ChildSignatures and Characters.
// Each Compartment consists of Indexes + Row Structures
class CortexCacheCompartment{

public:

	// Returns the name of the Temp File(s) which are based upon the Signature/CharCode.
	// It will calculate this value so that storage is not required within the parent vector.
	// Every ChildSignature has 2 files associated with it.  The "IndexFile" and the "RowContainer".
	std::string virtual getFileNameForIndexes() = 0;

	// The Row Container is separated to minimize "memory marshalling" as new rows are added.
	// The Rows can be added to the bottom (like an auto-increment ID) so there is work to do when the indexes are re-arranged.
	std::string virtual getFileNameForRows() = 0;

	// Why ask for the filename when you could get an Fstream Pointer that is already opened for you.
	std::fstream& getFileStreamReferenceForIndexes();
	std::fstream& getFileStreamReferenceForDataRows();

	// For "CortexCacheCharacter", it will return the number of "Child Signatures" contained under the characters.
	// For "CortexCacheChildSignature" it will return the number or rows.
	// It does this by looking at the file size of the "Cache Files"... since other processes may have been writing to them.
	long getNumberOfElementsInCacheFiles();

	// It would have been nice to initialize the File Handles within the Base Constructur.
	// However, Base Class can't figure out the "File Name Values" which must be defined within the derrived classes.
	void static initializeFileHandles(std::fstream& fsIndexes, const std::string& fileNameIndexes, std::fstream& fsDataRows, const std::string& fileNameDataRows, std::fstream& fsMasterFile, const std::string& fileNameMaster);

	// The base constructor.
	CortexCacheCompartment();

	// For Child Objects, the "Parent Container" can infer the Character Code without this member.
	// However, for autonomy, of the "filename methods", we need to know what Character Code it is working with.
	std::size_t characterCode;

protected:

	std::string indexFileNameCache;
	std::string rowFileNameCache;

	std::size_t totalCharactersInMaster;

	void virtual extractDataFromMaster() = 0;

	std::fstream fileStreamIndexes;
	std::fstream fileStreamDataRows;
	std::fstream fileStreamMasterFile;
};


// ------ FIRST Compartment -------------
// This container will be held within a Vector...
// ... which will be held inside of an object that is itself held in a vector.
class CortexCacheChildSignature : public CortexCacheCompartment
{

public:

	// The constructor requires a CharacterCode and a ChildSignature which belongs to it.
	// Make sure that you only create one object per (character + ChildSig) or it will complain (because there will already be a TempFile on disk).
	// The constructor will extract all of the "C27 Indexes" and "RowValues" from the "MASTER FILE".
	// If it can't find any "ChildIndexes" (belonging to the Character Code), it will create the TempFile anyway.
	// This is necessary because you can add new ChildSignatures to the database before the Master file is updated.
	// totalChildSignatures is going to be assigned to our "Row Count" member variable.  It just saves us a little work from looking it up ourself.
	CortexCacheChildSignature(std::size_t charCode, std::size_t totalCharCodes, SignatureStruct childSignature, long totalChildSignatures);

	// Will delete both TempFiles.
	~CortexCacheChildSignature();

	// Inherited Virtual Methods
	std::string virtual getFileNameForIndexes();
	std::string virtual getFileNameForRows();

	// Add 1 for the NULL Byte
	// The Vectors (in parent compartment) will sort entries alphabetically based upon this signature.
	// It needs to be made public so that the Vector can access the member.
	char recordName[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];

	// While declared "public" in the base class, it needs to be re-declared in the derived class to be public.
	std::size_t characterCode;

	// These are stored parallel to the extracted structure and will remain solely in memory until the program decides to consolidate the temp files.
	// Cortex::ChildSignature.byteTalley & Cortex::ChildSignature.rowCount
	std::streamsize byteTalley;
	long rowCount;

private:

	// This will be set from the constructor.
	// It will save us time from having to look up the value when we are going locate the "Child Signature" within the Index.
	long totalChildSignaturesInMaster;

	// This is called from the constructor once, but it seems cleaner to put it in a separate method.
	void virtual extractDataFromMaster();

	// Make default Constructor private.
	CortexCacheChildSignature();
	CortexCacheChildSignature(CortexCacheChildSignature const&);
	void operator=(CortexCacheChildSignature const&);
};


// ------ SECOND Compartment (builds off of 1st) -------
// This container will be held within a Vector.
// Each Container gets to contain 1 CharacterCode ... plus an array of ChildSignatures.
class CortexCacheCharacter : public CortexCacheCompartment
{
public:

	// Notice we don't need the "Character Code" because it is stored in this object.
	// This "cache compartment" will hold a Vector of ChildSignatures.
	// The Master Interface Class "CortexCache" does not have access to the the "ChildSignatures Objects"... it has to get them indirectly through this class.
	// If the "Child Signatures" have not been cached (and stored within the member vector), these methods will initiate the process transparently.
	std::string getFile_Row_Indexes(SignatureStruct childSignature);
	std::string getFile_Row_Data(SignatureStruct childSignature);

	std::fstream& getStream_Row_Indexes(SignatureStruct childSignature);
	std::fstream& getStream_Row_Data(SignatureStruct childSignature);

	// The constructor requires a CharacterCode.
	// Make sure that you only create one object per character or it will complain (because there will already be a TempFile on disk).
	// The constructor will extract the Character and "ChildIndexes" from the "MASTER FILE".
	// The CharacterCode must be found within the MASTER file or else it will throw an Exception.
	// Character codes must be defined ahead of time.  You can not add more during runtime.
	CortexCacheCharacter(std::size_t charCode, std::size_t totalCharCodes);

	// We need to know if Character/ChildSignatures are cached at the time of consolidation.
	bool isChildSignatureCached(SignatureStruct childSignature);

	// Will delete both TempFiles.
	~CortexCacheCharacter();

	// Inherited Virtual Methods
	std::string virtual getFileNameForIndexes();
	std::string virtual getFileNameForRows();


	// While declared "public" in the base class, it needs to be re-declared in the derived class to be public.
	std::size_t characterCode;

	// These are stored parallel to the extracted structure and will remain solely in memory until the program decides to consolidate the temp files.
	// Cortex::Character.byteTalley & Cortex::Character.childSignatureCount
	std::streamsize byteTalley;
	long childSignatureCount;

	// Each CacheContainer will contain an Array of "ChildSignatures" which have been extracted into the appropriate temp files.
	// By defining the size of the CortexCacheSignatures using C-strings... we can iterate through the list quickly (or use Sort Algorithm) not worring about using pointers... new/delete, etc.
	// The CortexCacheCharacter class is heirarchially 1 level higher than CortexCacheChildSignature (even though they inherit from the same Base Class).
	std::vector<CortexCacheChildSignature*> childSignatureCacheObjects;

	// This will use a "Binary Search" algorithm on the Vector stored within this object.
	// This method is being piggy backed to get the Index of a record which is already cached, and also to figure out the index location where a new item should be inserted.
	// Read the comments on the method definition.
	long getIndexPositionForInsertionOfChildSignature(SignatureStruct childSignature);

private:

	// Will pull the "C27/LX Indexes" and "Row Structures" out of the master file.
	// It does not hurt to call this multiple times.  It will only cache the temp files one time for the duration of the program.
	void cacheChildSignature(SignatureStruct childSignature);

	// This is called from the constructor once, but it seems cleaner to put it in a separate method.
	void virtual extractDataFromMaster();

	// Make default Constructor private.
	CortexCacheCharacter();
	CortexCacheCharacter(CortexCacheCharacter const&);
	void operator=(CortexCacheCharacter const&);
};

