#include <iostream>
#include <string>
#include <sstream>
#include <vector>
#include <fstream>
#include "SignatureStruct.h"
#include "TriangleColumns.h"
#include "QueryStruct.h"
#include "Cortex.h"
#include "CortexSearch.h"
#include "CortexCacheCompartment.h"


// Base constructor
CortexCacheCompartment::CortexCacheCompartment(){

	this->totalCharactersInMaster = 0;
	this->characterCode = 0;

	this->indexFileNameCache = "";
	this->rowFileNameCache = "";


}


// This method is inheritted for both Cache Compartments.
// Since the file handles are initialized within the Constructor, these file handles OK to use (without openining, etc.).
std::fstream& CortexCacheCompartment::getFileStreamReferenceForIndexes()
{
	return this->fileStreamIndexes;

}
std::fstream& CortexCacheCompartment::getFileStreamReferenceForDataRows()
{
	return this->fileStreamDataRows;
}



void CortexCacheCompartment::initializeFileHandles(std::fstream& fsIndexes, const std::string& fileNameIndexes, std::fstream& fsDataRows, const std::string& fileNameDataRows, std::fstream& fsMasterFile, const std::string& fileNameMaster)
{
	// If there were any Temp Files left over from last time... Oh, well.
	std::remove(fileNameIndexes.c_str());
	std::remove(fileNameDataRows.c_str());

	// The fstream object does not create new files like ofstream does.
	std::ofstream createFile;
	createFile.open(fileNameIndexes);
	createFile.close();
	createFile.open(fileNameDataRows);
	createFile.close();

	// Now open all 3 of our file handles for both of our Derived classes.
	fsIndexes.open(fileNameIndexes);
	if(!fsIndexes.is_open()){
		throw "Error oppening Temp Index File";
	}

	fsDataRows.open(fileNameDataRows);
	if(!fsDataRows.is_open()){
		throw "Error oppening Temp Data Rows File";
	}

	fsMasterFile.open(Cortex::Globals::CORTEX_MASTER_BINARY_FILE);
	if(!fsMasterFile.is_open()){
		throw "Error oppening Master Binary File";
	}

std::cout << "Initializing BASE constructor.\nIndexes: " << fileNameIndexes << " \nRows: " << fileNameDataRows << "\n";
}

// This will return the number of "Child Signatures" or the number or "Rows".
// This method is inherited by both "Cache Compartments".
long CortexCacheCompartment::getNumberOfElementsInCacheFiles()
{
	// Regardless of the type of indexes that are stored, the Structure size is the same.
	std::streamsize fileSizeIndexes = CortexSearch::getFileSize(this->fileStreamIndexes);

	if(fileSizeIndexes % (sizeof Cortex::Index) != 0){
		throw "Incorrect file size for indexes files in method CortexCacheCompartment::getNumberOfElementsInCacheFiles";
	}

	return static_cast<long>(fileSizeIndexes / (sizeof Cortex::Index));
}



// Constructor for First derived class.
CortexCacheChildSignature::CortexCacheChildSignature(std::size_t charCode, std::size_t totalCharCodes, SignatureStruct childSignature, long totalChildSignatures)
{
	if(childSignature.digestedStr.size() < Cortex::Globals::CORTEX_SIGNATURE_LENGTH){
		throw "The child signature has not been set";
	}

	// Make sure there is a Null Character for the 29 position.
	std::memset(this->recordName, 0, sizeof this->recordName);

	// Copy the std::string from the contructor argument into the C-String member
	strncpy(this->recordName, childSignature.digestedStr.c_str(), Cortex::Globals::CORTEX_SIGNATURE_LENGTH);

	// We won't know this until we fetch the value out of the "Child Signature Struct" in the master.
	this->rowCount = 0;

	this->totalCharactersInMaster = totalCharCodes;
	this->totalChildSignaturesInMaster = totalChildSignatures;

	CortexCacheCompartment::initializeFileHandles(this->fileStreamIndexes, this->getFileNameForIndexes(), this->fileStreamDataRows, this->getFileNameForRows(), this->fileStreamMasterFile, Cortex::Globals::CORTEX_MASTER_BINARY_FILE);

	this->extractDataFromMaster();
}


// Constructor for Second derived class.
CortexCacheCharacter::CortexCacheCharacter(std::size_t charCode, std::size_t totalCharCodes){

	// Just in case there is any dangling memory.
	this->childSignatureCacheObjects.clear();

	this->totalCharactersInMaster = totalCharCodes;

	CortexCacheCompartment::initializeFileHandles(this->fileStreamIndexes, this->getFileNameForIndexes(), this->fileStreamDataRows, this->getFileNameForRows(), this->fileStreamMasterFile, Cortex::Globals::CORTEX_MASTER_BINARY_FILE);

	this->extractDataFromMaster();
}



// The Base Class can't return different filenames for both of the classes because the "Record Names" don't exist for one of the compartments.
// The Base Class made these both "pure virtual" methods.
std::string CortexCacheChildSignature::getFileNameForIndexes(){

	// This may save a little overhead so that the "string stream" object doesn't have to fire up for every request.
	if(this->indexFileNameCache.size() == 0){
		this->indexFileNameCache = "i-" + Cortex::Globals::getCharCodeString(this->characterCode) + "-" + this->recordName + ".tmp";
	}
	return this->indexFileNameCache;
}
std::string CortexCacheChildSignature::getFileNameForRows(){
	if(this->rowFileNameCache.size() == 0){
		this->rowFileNameCache = "r-" + Cortex::Globals::getCharCodeString(this->characterCode) + "-" + this->recordName + ".tmp";
	}
	return this->rowFileNameCache;
}
//-----------------------------------------------
std::string CortexCacheCharacter::getFileNameForIndexes(){
	if(this->indexFileNameCache.size() == 0){
		this->indexFileNameCache = "i-" + Cortex::Globals::getCharCodeString(this->characterCode) + "-C.tmp";
	}
	return this->indexFileNameCache;
}
std::string CortexCacheCharacter::getFileNameForRows(){
	if(this->rowFileNameCache.size() == 0){
		this->rowFileNameCache = "r-" + Cortex::Globals::getCharCodeString(this->characterCode) + "-C.tmp";
	}
	return this->rowFileNameCache;
}



// This will use the Binary Search algorithm to find a new Index location for the "Child Signature" within the member-vector.
// It will also tell you the index location of an existing record.
// So how do you know the difference??? When you get back the result, check to see if the signatures match between the Needle and the RecordName within the vector.
//    * If the signatures match, then it is already in cache.
//    * If the signature don't match up, then you have your index number where it should be inserted.
long CortexCacheCharacter::getIndexPositionForInsertionOfChildSignature(SignatureStruct signature)
{
	long left = 0;
	long right = this->childSignatureCacheObjects.size();

	// Don't access empty vector.
	if(right == 0){
		return 0;
	}

	while(left <= right){

		long middle = (left + right) / 2;

		if(std::strcmp(this->childSignatureCacheObjects.at(middle)->recordName, signature.digestedStr.c_str()) == 0){
			return middle;
		}
		else if (std::strcmp(this->childSignatureCacheObjects.at(middle)->recordName, signature.digestedStr.c_str()) > 0){
			right = middle - 1;
		}
		else{
			left = middle + 1;
		}
	}

	if(left < 0){
		return 0;
	}
	else{
		return left;
	}
}




// Write all of the "Database Rows" in 1 file, and all of the "C27/LX Indexes) in the second file belonging to the combination of CharacterCode/ChildSignature
void CortexCacheChildSignature::extractDataFromMaster(){

	// Copy the Cstring into a "SignatureStruct" for the method call.
	SignatureStruct childSigSearch;
	childSigSearch.digestedStr = this->recordName;

	// Figure out the ByteOffset of the "Child Signature Structure" relative to the beggining of the MasterBinaryFile.
	std::streamsize masterByteOffset = (sizeof Cortex::Header);
	masterByteOffset += (sizeof Cortex::Index) * this->totalCharactersInMaster;
	masterByteOffset += CortexSearch::getByteOffsetFromCharacterIndex(this->fileStreamMasterFile, this->characterCode, this->totalCharactersInMaster);
	masterByteOffset += (sizeof Cortex::Character);

	// This method call will throw an exception if the "Child Signature" can not be located within the index.
	masterByteOffset += CortexSearch::getByteOffsetFromChildSignatureIndex(this->fileStreamMasterFile, this->characterCode, this->totalCharactersInMaster, childSigSearch, this->totalChildSignaturesInMaster);

	Cortex::ChildSignature cortexChildSig={0};

	this->fileStreamMasterFile.seekg(masterByteOffset, std::ios::beg);
	this->fileStreamMasterFile.read((char*)&cortexChildSig, (sizeof Cortex::ChildSignature));

	if(!this->fileStreamMasterFile){
		throw "Error reading the file handle in method CortexCacheChildSignature::extractDataFromMaster.";
	}

	if(cortexChildSig.characterCode != this->characterCode){
		throw "The index appears to be corrupted in method CortexCacheChildSignature::extractDataFromMaster for the Character Code";
	}

	if(strcmp(cortexChildSig.recordName, this->recordName) != 0){
		throw "The index appears to be corrupted in method CortexCacheChildSignature::extractDataFromMaster when comparing the Child Signatures.";
	}


	// There are only 2 members within the "Character Structure".  We are going to store these values in our object for the lifetime of the program.
	// If someone changes the TempFile (they can only add stuff)... then we will need to increment these values based upon the size of the TempFile.
	this->byteTalley = cortexChildSig.byteTalley;
	this->rowCount = cortexChildSig.rowCount;


	// --------------------- Now fetch all of the "C27 / LX Indexes" ---------------

	// Get all of the indexes from the Master file in a vector.
	// There are going to be 28 collections of indexes.  Each of the collections are stacked on top of each other (with an equal count) based upon the priority in the "Triangle" class.
	std::vector<Cortex::Index> c27andLxIndexesVect;
	CortexSearch::getIndexesVector(this->fileStreamMasterFile, c27andLxIndexesVect, masterByteOffset, (this->rowCount * Cortex::Globals::NUMBER_OF_INDEXED_COLUMNS_ON_DATA_ROWS));

	// Write them into the temp.  That is all that this temp file will get.
	std::streamsize tempFileOffset = 0;
	for(unsigned long x=0; x < c27andLxIndexesVect.size(); x++){
		tempFileOffset = Cortex::Write::toBinFile(this->fileStreamIndexes, c27andLxIndexesVect[x], tempFileOffset);
	}

	// This will position the "master byte offset" at the bottom of all "Child Signature Indexes" where the very first "Child Structure Starts".
	masterByteOffset += (sizeof Cortex::Index) * c27andLxIndexesVect.size();

	// Because we don't have any "Byte Offsets" to skip over, we don't have to check the gaps between our Indexes.
	// Therefore, let's free up some RAM now that our Indexes are stored in the TempFile.
	c27andLxIndexesVect.clear();


	// ---------------------- Now extract all of the "Row Structures" which fall immediately underneath the Indexes. -------------------------

	// Extract all of the Row Structures into a Vector first so that the disk doesn't thrash as much.
	std::vector<Cortex::Row> rowVect;

	for(long x=0; x < this->rowCount; x++){

		// All of the "Row Structures" are stored contiguously with the same POD size.
		Cortex::Row rowStructure={0};
		masterByteOffset = Cortex::Read::fromBinFile(this->fileStreamMasterFile, rowStructure, masterByteOffset);

		// This will make a copy of the Index Structure... so don't worry about it going out of scope.
		rowVect.push_back(rowStructure);
	}

	// Output the Row Structures to their very own temp file (relative to the "Child Signature Structure" [which is relative to the "Character Stucture"]).
	tempFileOffset = 0;
	for(unsigned long x=0; x < rowVect.size(); x++){
		tempFileOffset = Cortex::Write::toBinFile(this->fileStreamDataRows, rowVect[x], tempFileOffset);
	}

	// Now the rowVect can go out of scope.
}

// Write all of the "Child Structures" in 1 file and all of the "Child Signatures" in the second.
void CortexCacheCharacter::extractDataFromMaster(){

	// --------------------- Get the "Character Stucutre" out of the master (will Tell us out many Child Signatures it has) ---------------

	// Figure out the ByteOffset of the CharacterCode Structure relative to the beggining of the MasterBinaryFile.
	std::streamsize masterByteOffset = (sizeof Cortex::Header);
	masterByteOffset += (sizeof Cortex::Index) * this->totalCharactersInMaster;
	masterByteOffset += CortexSearch::getByteOffsetFromCharacterIndex(this->fileStreamMasterFile, this->characterCode, this->totalCharactersInMaster);

	Cortex::Character cortexChar={0};

	// Don't skip ahead by the "Byte Talley"... just advance past the "Character Stucture" where we will find the "Child Signature Indexes".
	masterByteOffset = Cortex::Read::fromBinFile(this->fileStreamMasterFile, cortexChar, masterByteOffset);

	if(cortexChar.characterCode != characterCode){
		throw "The index appears to be corrupted in method CortexCacheCharacter::extractDataFromMaster";
	}

	// There are only 2 members within the "Character Structure".  We are going to store these values in our object for the lifetime of the program.
	// If someone changes the TempFile (they can only add stuff)... then we will need to increment these values based upon the size of the TempFile.
	this->byteTalley = cortexChar.byteTalley;
	this->childSignatureCount = cortexChar.childSignatureCount;


	// --------------------- Now the Child Signatures Indexes ---------------

	// Get them from the master in a vector.
	std::vector<Cortex::Index> childSigIndexesVect;
	CortexSearch::getIndexesVector(this->fileStreamMasterFile, childSigIndexesVect, masterByteOffset, this->childSignatureCount);

	// Write them into the temp.  That is all that this temp file will get.
	std::streamsize tempFileOffset = 0;
	for(unsigned long x=0; x < childSigIndexesVect.size(); x++){
		tempFileOffset = Cortex::Write::toBinFile(this->fileStreamIndexes, childSigIndexesVect[x], tempFileOffset);
	}

	// This will position the "master byte offset" at the bottom of all "Child Signature Indexes" where the very first "Child Structure Starts".
	masterByteOffset += (sizeof Cortex::Index) * childSigIndexesVect.size();



	// ---------------------- Now the Child Structures -------------------------

	// To get the Child Structures we have to skip over the Row Containers.
	// That's easy to do because the Index Vectors conain the "Byte Offsets" where we can find them... and also there are "Byte Talleys" stored on the stuctures themselves.
	std::vector<Cortex::ChildSignature> childSigVect;
	Cortex::ChildSignature childSig={0};

	// We are going to make sure that the indexes are not out of sync with the actual structures.
	std::streamsize childSigByteTalleyCounter = 0;

	for(long x=0; x < this->childSignatureCount; x++){

		masterByteOffset = Cortex::Read::fromBinFile(this->fileStreamMasterFile, childSig, masterByteOffset);

		// This will make a copy of the Index Structure... so don't worry about it going out of scope.
		childSigVect.push_back(childSig);

		// The first "Child Signature Index" will show a "Byte Offset" of 0.
		if(childSigByteTalleyCounter != childSigIndexesVect[x].byteOffset){
			throw "The Byte Talleys don't match up to the ByteOffsets in the index";
		}

		// This will skip ahead of the "Row Structures" and "Row Indexes" for the next time we read a structure.
		// This is relative to the begginning of the file.
		masterByteOffset += childSig.byteTalley;

		// This will increment a variable used to make sure that our Indexes match up to the structure sizes.
		childSigByteTalleyCounter += childSig.byteTalley;
	}

	// Now write all of the "Child Stuctures" to there very own file (relative to the main Character Code).
	tempFileOffset = 0;
	for(unsigned long x=0; x < childSigVect.size(); x++){
		tempFileOffset = Cortex::Write::toBinFile(this->fileStreamDataRows, childSigVect[x], tempFileOffset);
	}

	// Now both of the vectors that we used can go out of scope.
	// Another process can extract the temp files and store them in RAM if they want.
}


void CortexCacheCharacter::cacheChildSignature(SignatureStruct childSignature)
{
	// Don't cache it twice.
	if(this->isChildSignatureCached(childSignature)){
		return;
	}

	// Now that we know the signature is not cached, this will give us the record for insertion.
	unsigned long locForChildSigInsertInVector = this->getIndexPositionForInsertionOfChildSignature(childSignature);

	// Allocate a Cache Compartment on the heap.
	// Our destructor will clean it up... which is kind of uncessary since this is a singleton and this class will only continue to add additional objects during the program lifetime.
	CortexCacheChildSignature* childSigCacheObj = new  CortexCacheChildSignature(this->characterCode, this->totalCharactersInMaster, childSignature, this->childSignatureCount);

	// Now we can insert the new "Child Signature Cache Pointer Object" into our vector at the proper index location.
	std::vector<CortexCacheChildSignature*>::iterator insertIt = this->childSignatureCacheObjects.begin() + locForChildSigInsertInVector;
	insertIt = this->childSignatureCacheObjects.insert(insertIt, childSigCacheObj);
}



CortexCacheCharacter::~CortexCacheCharacter()
{
	// Clean up the heap for pointers stored in the Vector.
	for(unsigned long i=0; i < this->childSignatureCacheObjects.size(); i++){
		delete this->childSignatureCacheObjects.at(i);
	}

	// Remove TempFiles when object(s) goes out of scope (which shoudl be for the program lifetime).
	std::remove(this->getFileNameForIndexes().c_str());
	std::remove(this->getFileNameForRows().c_str());
}

// The final cache compartment doesn't have any objects in a vector to delete.
CortexCacheChildSignature::~CortexCacheChildSignature(){
	std::remove(this->getFileNameForIndexes().c_str());
	std::remove(this->getFileNameForRows().c_str());
}




// It is indirectly calls a method from the 1st Cache Compartment inside of a vector.
// These method themselves are called from within a Vector, but in the main "CortexCache" interface.
std::string CortexCacheCharacter::getFile_Row_Indexes(SignatureStruct childSignature)
{
	this->cacheChildSignature(childSignature);
	long cachedIndexLoc = this->getIndexPositionForInsertionOfChildSignature(childSignature);
	return this->childSignatureCacheObjects.at(cachedIndexLoc)->getFileNameForIndexes();
}
std::string CortexCacheCharacter::getFile_Row_Data(SignatureStruct childSignature)
{
	this->cacheChildSignature(childSignature);
	long cachedIndexLoc = this->getIndexPositionForInsertionOfChildSignature(childSignature);
	return this->childSignatureCacheObjects.at(cachedIndexLoc)->getFileNameForRows();
}
// ----------------  Same thing, but for File Streams instead of File Names -----------
std::fstream& CortexCacheCharacter::getStream_Row_Indexes(SignatureStruct childSignature)
{
	this->cacheChildSignature(childSignature);
	long cachedIndexLoc = this->getIndexPositionForInsertionOfChildSignature(childSignature);
	return this->childSignatureCacheObjects.at(cachedIndexLoc)->getFileStreamReferenceForIndexes();
}
std::fstream& CortexCacheCharacter::getStream_Row_Data(SignatureStruct childSignature)
{
	this->cacheChildSignature(childSignature);
	long cachedIndexLoc = this->getIndexPositionForInsertionOfChildSignature(childSignature);
	return this->childSignatureCacheObjects.at(cachedIndexLoc)->getFileStreamReferenceForDataRows();
}



// This is called from the Main Cache Interface through a Vector Element.
bool CortexCacheCharacter::isChildSignatureCached(SignatureStruct childSignature)
{
	unsigned long locationOfChildSigInVector = this->getIndexPositionForInsertionOfChildSignature(childSignature);

	// The location for insertion could be greater than the number of elements.
	// For example.  The first element will have an "index for insertion" of Zero... but the size will be zero.
	// Therefore we can't access the [0] element.
	// If we can find equality for the Character Code, then we know it is cached already (so we can return).
	if(locationOfChildSigInVector < this->childSignatureCacheObjects.size()){
		if(strcmp(this->childSignatureCacheObjects[locationOfChildSigInVector]->recordName, childSignature.digestedStr.c_str()) == 0){
			return true;
		}
	}

	return false;
}



