#pragma once

// The CortexCache class (and CacheCompartment Classes) are used to extract and organize "Temporary Cache Files" on disk relative to the MASTER.
// It job description may include re-arranging indexes and sorting, but it should not be used to query the database.
// Its main mission is to pass filenames back to the BrainIO class so it has smaller pieces to work with.  The database may grow too large to keep everything in RAM.



// --------- This is the Main Interface ---------------
// The other 2 compartments cache the indexes with TempFiles.
// This Master interface will cache all of the CharacterCodes & Character Structures completely in memory.
// That isn't a big deal, even someone used every character in the unicode character set.
// CortexCacheCharacter will write all of the "ChildSignatureIndexes" to a Tempfile (with 2 filnames based upon the Character Code).
// CortexCacheChildSignature will write all of the "RowSignatures" to a TempFile (with 2 filenames based upon the Charcter Code & and the "Child Signature").
// There are no "Character Stuctures" to write to disk because the master starts out with all 65K characters.
//		- CortexCacheChildSignature will write all of the "Database Rows" in 1 file, and all of the "C27/LX Indexes) in the second file.
//		- CortexCacheCharacter will write all of the "Child Structures" in 1 file and all of the "Child Signatures" in the second.
class CortexCache {

public:

	// Singleton Pattern
	static CortexCache& getInstance()
	{
		static CortexCache instance;
		return instance;
	}


	// This will use a "Binary Search" algorithm, based upon the entries within the Vector.
	// The vector that is searched within is in this class.
	bool isCharacterCached(std::size_t characterCode);

	// Will return false if the Character Has not already been cached.  That has to happen before a Child Signature can be cached.
	// First it locates the CortexCacheCharacter (in Vector Cache), then it calls isChildSignatureCached() on that particular object.
	bool isChildSignatureCached(std::size_t characterCode, SignatureStruct childSignature);


	// The following 2 Cache Compartments have 2 files associated with them.
	// 1) The Indexes (which need to be completely re-written when something is inserted in the middle).
	// 2) The Data (new elements will be inserted to the bottom.

	// These method cache the character/signature files (if they haven't already been initialized).
	std::string getFile_ChildSignature_Indexes(std::size_t characterCode);
	std::string getFile_ChildSignature_Data(std::size_t characterCode);
	// -----------
	std::string getFile_Row_Indexes(std::size_t characterCode, SignatureStruct childSignature);
	std::string getFile_Row_Data(std::size_t characterCode, SignatureStruct childSignature);


	// Same thing but returns an fStream pointer which is already open and ready to use.
	std::fstream& getStream_ChildSignature_Indexes(std::size_t characterCode);
	std::fstream& getStream_ChildSignature_Data(std::size_t characterCode);
	// -----------
	std::fstream& getStream_Row_Indexes(std::size_t characterCode, SignatureStruct childSignature);
	std::fstream& getStream_Row_Data(std::size_t characterCode, SignatureStruct childSignature);


	// This will find out the lastest size on disk by seeking to the end of the Cache File and determining the file size.
	// So, don't worry about this method call returning stale results.
	long getCurrentChildSignatureIndexCount(std::size_t characterCode);
	long getCurrentRowDataIndexCount(std::size_t characterCode, SignatureStruct childSignature);


	// Will delete memory on Heap allocated to the vector.
	~CortexCache();


	// This will gather all of the "TempFiles" and merge the data back into the TempFile.
	void consolidate();

private:

	// If the character code is already cached in the Vector, it will contain the index position where it can be found.
	// If the Character Code is not cached yet, it will return the position where it is to be inserted.
	// How do you know the difference?  Compare the element you are searching for against the value in the vector with the return value.
	std::size_t getIndexPositionForInsertionOfCharacterCode(std::size_t searchCharacterCode);

	// We are going to "lazy load" the characters out of the Master File.
	// We can hold up to the entire Unicode library in Cache.
	// The size of this vector will let us know how many we have cached already.
	// These vectors will be sorted numerically based upon "Character Code" so that the records may be located quickly with a "binary search" algorithm.
	// Because we are storing pointers to "new" objects, the vector has quicker "random write" performance.
	// If we get back a match from a "sort", we need to create a character Object and store it in the vector for later.
	std::vector<CortexCacheCharacter*> characterCacheObjects;

	// Will pull the "Character" and "ChildIndexes" out of the master file.
	// Will throw an exception if the Character Code can not be located within the Master file.
	// It does not hurt to call this multiple times.  It will only cache the temp files one time for the duration of the program.
	void cacheCharacter(std::size_t characterCode);

	// Will fill up an Index by reference so that you don't have to use "new".
	// Basically just loads up all of the ChildSignatureIndexes from the Temp file.
	// This method will fail if the "Character Code" is not already cached.
	void getLatestChildIndexes(std::vector<Cortex::Index>& childSigIndexesVect, std::size_t characterCode);

	// Same thing as getLatestChildIndexes, but a level deeper.
	// This will fail if the either the "Character Code" or the "Child Signature" has not been cached already.
	void getLatestRowIndexes(std::vector<Cortex::Index>& rowIndexesVect, std::size_t characterCode, SignatureStruct childSignature);

	// We don't need a "Replace Byte Offset" because the "Row Structures" are always the same size, which means we can calculate it based upon the "Element Position" (which is like an Auto-increment ID).
	// Since the "Replace Byte Offset" can be calculated, that means that we can keep the number of method parameters to magic #6 or below.  Whoaah, that was a close one :)
	// Actually, only 1 "Search Parameter" is required, be we woudl like more to verify that there is no corruption.
	void searchReplaceRowIndexPosistions(std::vector<Cortex::Index>& rowIndexesVect, long searchIndexPos, long replaceIndexPos, long searchElementPos, long replaceElementPos, std::streamsize searchByteOffset);

	// This value will not change.
	// We will only fetch the value once out of the Master file.
	std::size_t totalCharactersInMaster;

	// Make private for the Singleton Pattern.
	CortexCache();

	std::fstream fileStreamMasterFile;

	// Make the Copy Constructor and the Copy Opertator inaccessable.
	CortexCache(CortexCache const&);
	void operator=(CortexCache const&);

};