#include <iostream>
#include <string>
#include <sstream>
#include <vector>
#include <fstream>
#include "SignatureStruct.h"
#include "TriangleColumns.h"
#include "QueryStruct.h"
#include "Cortex.h"
#include "CortexSearch.h"
#include "CortexCopy.h"
#include "CortexCacheCompartment.h"
#include "CortexCache.h"


// Constructor for the master interface.
CortexCache::CortexCache()
{
	this->characterCacheObjects.clear();

	this->fileStreamMasterFile.open(Cortex::Globals::CORTEX_MASTER_BINARY_FILE);
	if(!this->fileStreamMasterFile.is_open()){
		throw "Error oppening Master Binary File in CortexCache constructor.";
	}

	// In English, most people will not use more than 100 character codes.
	// Asian languages may use considerablly more.
	this->characterCacheObjects.reserve(500);

	this->totalCharactersInMaster = 0;
}

long CortexCache::getCurrentChildSignatureIndexCount(std::size_t characterCode)
{
	std::fstream& fileH = this->getStream_ChildSignature_Indexes(characterCode);
	std::streamsize fileSize = CortexSearch::getFileSize(fileH);

	if(fileSize % (sizeof Cortex::Index) != 0){
		throw "There appears to be index corruption in method getCurrentChildSignatureIndexCount";
	}

	return static_cast<long>(fileSize / (sizeof Cortex::Index));
}

long CortexCache::getCurrentRowDataIndexCount(std::size_t characterCode, SignatureStruct childSignature)
{
	std::fstream& fileH = this->getStream_Row_Indexes(characterCode, childSignature);
	std::streamsize fileSize = CortexSearch::getFileSize(fileH);

	if(fileSize % (sizeof Cortex::Index) != 0){
		throw "There appears to be index corruption in method getCurrentRowDataIndexCount";
	}

	return static_cast<long>(fileSize / (sizeof Cortex::Index));
}

// This works just like getIndexPositionForInsertionOfChildSignature().
// It is used for both New/Existing index positions.
// There is always a better way to abstract "logic in common"... so sue me, I don't have any money :).
std::size_t CortexCache::getIndexPositionForInsertionOfCharacterCode(std::size_t searchCharacterCode){

	std::size_t left = 0;
	std::size_t right = this->characterCacheObjects.size();

	// Don't access empty vector.
	if(right == 0){
		return 0;
	}

	while(left <= right){

		std::size_t middle = (left + right) / 2;

		if(this->characterCacheObjects.at(middle)->characterCode == searchCharacterCode){
			return middle;
		}
		else if(this->characterCacheObjects.at(middle)->characterCode > searchCharacterCode){
			right = middle - 1;
		}
		else{
			left = middle + 1;
		}
	}

	if(left < 0){
		return 0;
	}
	else{
		return left;
	}
}

// This is the from the Main Interface.  It will store the "Character Indexes" in memory, rather than writing out to a temp file.
// You can call this method repeatedly and and it will only extract the character code into memory once.
void CortexCache::cacheCharacter(std::size_t characterCode)
{
	// Don't cache it twice.
	if(this->isCharacterCached(characterCode)){
		return;
	}

	// Now that we know the CharacterCode is not cached, this will give us the record for insertion.
	std::size_t locForCharacterInsertInVector = this->getIndexPositionForInsertionOfCharacterCode(characterCode);

	// Get the number of characters out of the master file, then record the data into our member for future calls to reduce Disk I/O.
	if(this->totalCharactersInMaster == 0){
		Cortex::Header cortexHeader = CortexSearch::getHeaderStructure(this->fileStreamMasterFile);
		this->totalCharactersInMaster = cortexHeader.characterCount;
	}

	// Allocate a Cache Compartment on the heap.
	// Our destructor will clean it up... which is kind of uncessary since this is a singleton and this class will only continue to add additional objects during the program lifetime.
	CortexCacheCharacter* charCacheObj = new  CortexCacheCharacter(characterCode, this->totalCharactersInMaster);

	// Now we can insert the new "Character Info" into our vector at the proper index location.
	std::vector<CortexCacheCharacter*>::iterator insertIt = this->characterCacheObjects.begin() + locForCharacterInsertInVector;
	insertIt = this->characterCacheObjects.insert(insertIt, charCacheObj);
}


CortexCache::~CortexCache()
{
	// Clean up the heap for pointers stored in the Vector.
	for(std::size_t i=0; i < this->characterCacheObjects.size(); i++){
		delete this->characterCacheObjects.at(i);
	}
}


// This is the "Main Interface".
// Asking for a file (based upon the parameters) will extract the relavent data from the master file before returning the file name.
// If the data has already been extracted, it will quickly return the Filename.

// ----------------- File Handles from 1st Compartment -----------
std::string CortexCache::getFile_Row_Indexes(std::size_t characterCode, SignatureStruct childSignature)
{
	this->cacheCharacter(characterCode);
	std::size_t cachedIndexLoc = this->getIndexPositionForInsertionOfCharacterCode(characterCode);
	return this->characterCacheObjects.at(cachedIndexLoc)->getFile_Row_Indexes(childSignature);
}
std::string CortexCache::getFile_Row_Data(std::size_t characterCode, SignatureStruct childSignature)
{
	this->cacheCharacter(characterCode);
	std::size_t cachedIndexLoc = this->getIndexPositionForInsertionOfCharacterCode(characterCode);
	return this->characterCacheObjects.at(cachedIndexLoc)->getFile_Row_Data(childSignature);
}
// ----------------- File Handles from 2nd Compartment -----------
std::string CortexCache::getFile_ChildSignature_Indexes(std::size_t characterCode)
{
	// Doesn't hurt do do this twice.
	this->cacheCharacter(characterCode);
	std::size_t cachedIndexLoc = this->getIndexPositionForInsertionOfCharacterCode(characterCode);
	return this->characterCacheObjects.at(cachedIndexLoc)->getFileNameForIndexes();
}
std::string CortexCache::getFile_ChildSignature_Data(std::size_t characterCode)
{
	this->cacheCharacter(characterCode);
	std::size_t cachedIndexLoc = this->getIndexPositionForInsertionOfCharacterCode(characterCode);
	return this->characterCacheObjects.at(cachedIndexLoc)->getFileNameForRows();
}

// ---------------------------------------------
// Now with File Pointers instead of File Names.
// ---------------------------------------------

// ----------------- File Handles from 1st Compartment -----------
std::fstream& CortexCache::getStream_Row_Indexes(std::size_t characterCode, SignatureStruct childSignature)
{
	this->cacheCharacter(characterCode);
	std::size_t cachedIndexLoc = this->getIndexPositionForInsertionOfCharacterCode(characterCode);
	return this->characterCacheObjects.at(cachedIndexLoc)->getStream_Row_Indexes(childSignature);
}
std::fstream& CortexCache::getStream_Row_Data(std::size_t characterCode, SignatureStruct childSignature)
{
	this->cacheCharacter(characterCode);
	std::size_t cachedIndexLoc = this->getIndexPositionForInsertionOfCharacterCode(characterCode);
	return this->characterCacheObjects.at(cachedIndexLoc)->getStream_Row_Data(childSignature);
}
// ----------------- File Handles from 2nd Compartment -----------
std::fstream& CortexCache::getStream_ChildSignature_Indexes(std::size_t characterCode)
{
	this->cacheCharacter(characterCode);
	std::size_t cachedIndexLoc = this->getIndexPositionForInsertionOfCharacterCode(characterCode);
	return this->characterCacheObjects.at(cachedIndexLoc)->getFileStreamReferenceForIndexes();
}
std::fstream& CortexCache::getStream_ChildSignature_Data(std::size_t characterCode)
{
	this->cacheCharacter(characterCode);
	std::size_t cachedIndexLoc = this->getIndexPositionForInsertionOfCharacterCode(characterCode);
	return this->characterCacheObjects.at(cachedIndexLoc)->getFileStreamReferenceForDataRows();
}




bool CortexCache::isCharacterCached(std::size_t characterCode)
{
	std::size_t locationOfCharacterInVector = this->getIndexPositionForInsertionOfCharacterCode(characterCode);

	// The location for insertion could be greater than the number of elements.
	// For example.  The first element will have an "index for insertion" of Zero... but the size will be zero.
	// Therefore we can't access the [0] element.
	// If we can find equality for the Character Code, then we know it is cached already (so we can return).
	if(locationOfCharacterInVector < this->characterCacheObjects.size()){
		if(this->characterCacheObjects[locationOfCharacterInVector]->characterCode == characterCode){
			return true;
		}
	}

	return false;
}


bool CortexCache::isChildSignatureCached(std::size_t characterCode, SignatureStruct childSignature){

	if(!this->isCharacterCached(characterCode)){
		return false;
	}

	// Now that we know it is cached, we can trust that the "Index for Insertion" belongs to a Vector entry.
	std::size_t locationOfCharacterInVector = this->getIndexPositionForInsertionOfCharacterCode(characterCode);

	return this->characterCacheObjects.at(locationOfCharacterInVector)->isChildSignatureCached(childSignature);
}

// The "meat" of the operation is letting the application work with smaller files during runtime.
// This is the "Potatoes" of the operation.  If we can't combine the cache files back together again, then we can't share our brain.
void CortexCache::consolidate()
{
	// We need to consolidate all of the TempFiles back into a single binary temp file.
	// If we don't encounter any errors, we can remove the "Master Binary File" and rename the "Temporary Binary File" to be the master.
	std::fstream fsTempBin;
	Cortex::Globals::resetFile(fsTempBin, Cortex::Globals::CORTEX_TEMPORARY_BINARY_FILE, true);

	// ---------- Transfer Cortex Header ------
	Cortex::Header cortexHeader = CortexSearch::getHeaderStructure(this->fileStreamMasterFile);

	std::streamsize byteOffsetForTemp = 0;
	std::streamsize byteOffsetForMaster = 0;

	// Move to the very beggining of the file and put the header in there.
	byteOffsetForTemp = Cortex::Write::toBinFile(fsTempBin, cortexHeader, byteOffsetForTemp);

	// The master is still syncronized with the Full-Temp.
	byteOffsetForMaster = byteOffsetForTemp;


	// ---------- Transfer Character Indexes ------

	// Write all of the Character Index out to the temp file after loading them into a vector first.
	// We don't have to worry about any positions changing (All Characters exist in the master for all unicode entries).
	// However, we may need to return to one of these vectors and update the "Byte Offset" depending on what has changed in our TempFiles.
	std::vector<Cortex::Index> charIndexesVect;
	CortexSearch::getIndexesVector(this->fileStreamMasterFile, charIndexesVect, byteOffsetForMaster, this->totalCharactersInMaster);

	for(std::size_t i=0; i<charIndexesVect.size(); i++){
		byteOffsetForTemp = Cortex::Write::toBinFile(fsTempBin, charIndexesVect[i], byteOffsetForTemp);
	}

	// The Master and Temp file should still be syncronized.  All that we did so far was output the Header and Character Indexes... no structures yet.
	byteOffsetForMaster = byteOffsetForTemp;


	// ---------- Transfer Character Structures ------

	// So we have all of the Indexes available in cache now.
	// As we loop through them we need to figure out if it is a character which has been cached (and in a temp file), or if it must be extracted from the master.
	for(std::size_t i=0; i<charIndexesVect.size(); i++){

		// The Byte Offset for the "Master File" will always match what we have stored in the "Master Indexes".
		// For the "Consolodated Temp File", we won't know the "Byte Offset" until after the character structure has been completely written (because other software was writing to the TempFiles).
		// That means that we will have to update the indexes retroactively.
		std::streamsize loop_ByteOffsetForMaster = byteOffsetForMaster + charIndexesVect[i].byteOffset;

		std::size_t loop_CharCode = std::size_t(charIndexesVect[i].recordName);

		if(this->isCharacterCached(loop_CharCode)){

			std::size_t vectLocForChar = this->getIndexPositionForInsertionOfCharacterCode(loop_CharCode);

			// Read "Character Structure" from main.
			Cortex::Character cortexChar={0};
			loop_ByteOffsetForMaster = Cortex::Read::fromBinFile(this->fileStreamMasterFile, cortexChar, loop_ByteOffsetForMaster);

			// Let's put the "Master File Pointer" at the bottom of the ChildSignatureIndexes.
			// There could be less ChildSignatureIndexes in the master compared to the merged indexes.
			loop_ByteOffsetForMaster += cortexChar.childSignatureCount * (sizeof Cortex::Index);

			// Remember this value so that after we update the indexes, we won't have to do any searching to update the "Byte Talley".
			std::streamsize charCodeStructureLocationInTemp = byteOffsetForTemp;

			// Write a single "Character Structure" to temp.
			byteOffsetForTemp = Cortex::Write::toBinFile(fsTempBin, cortexChar, byteOffsetForTemp);

			// Other processes could have been writing to the "Cache Files".  This will extract the latest "Child Signature Indexes" for the given Character Code into a vector and make sure that they are properly sorted.
			std::vector<Cortex::Index> latestChildSigIndexesVect;
			this->getLatestChildIndexes(latestChildSigIndexesVect, loop_CharCode);

			// Write all of the "Child Signature Indexes" to the Full-Temp File sequentially.
			// The previous method call "getLatestChildIndexes" already verified that the indexes are sorted correcly.
			for(unsigned long x=0; x < latestChildSigIndexesVect.size(); x++){
				byteOffsetForTemp= Cortex::Write::toBinFile(fsTempBin, latestChildSigIndexesVect[x], byteOffsetForTemp);
			}

			// Remember this value so that after we update the Rows, etc... we won't have to do any searching to update the "Byte Offsets" where the "Child Structures" are stored in the temp.
			// This location points underneath the given Character Code in the FullTemp... underneath the very last "Child Signature Index".
			std::streamsize childStructuresLocationInTemp = byteOffsetForTemp;

			// Now we have to merge the "Child Signature" Rows from all of the temp files with our master.
			// Copy all of the "ChildSignature Stuctures", "Rows Indexes", and "Rows" from the master into the FullTemp (but only if we have NOT cached them).
			// Any Child Signature Structure which has been cached will end be appended to the bottom.
			// That means that we are going to skip over "ChildSigs / Rows" in the master.  That means we have to update ALL of the indexes (as we are going along).
			// Instead of thrashing the disk to change the indexes as we proceed, just change the value within the Vector in the loop, then overwrite the indexes aftewards (the list size won't change).

			for(unsigned long x=0; x < latestChildSigIndexesVect.size(); x++){

				// We are going to re-organize the indexes, so extract this value before we overwrite it within the vector below.
				long elementPositionForChildSigStructureInCacheFile = latestChildSigIndexesVect[x].elementPosition;

				// The "Byte Offsets" are relative to where the "Child Signature Structure" begins (after the ChildSignature Indexes).
				// So that means that the first Element (in the index) should have a "Byte Offset" of Zero.
				// Our various "cache files" may have scattered the "Element Positions" relative to the "Index Positions".
				// We are going to syncronize them in the vector now (still to be written to the file though).
				latestChildSigIndexesVect[x].byteOffset = byteOffsetForTemp - childStructuresLocationInTemp;
				latestChildSigIndexesVect[x].indexPosition = x;
				latestChildSigIndexesVect[x].elementPosition = x;

				SignatureStruct childSigSearch={0};
				childSigSearch.digestedStr =  latestChildSigIndexesVect[x].recordName;

				if(this->isChildSignatureCached(loop_CharCode, childSigSearch)){

					long vectLocForChildSig = this->characterCacheObjects[vectLocForChar]->getIndexPositionForInsertionOfChildSignature(childSigSearch);

					// There are 2 Temp Files which must be copied into the Full Temp.  1) Row Indexes  2) Row Values
					// There is also a single "ChildSignatureStructure" that will be written before the stack of "Row Indexes".
					Cortex::ChildSignature childSigStruct={0};

					// The "Cache Files" for Child Signatures Data Rows do not have any Sub-structures within.  The C27/LX Rows have been extracted into another file.
					// Since the Child Signatures are stored contiguously in our Cache Files, we will use the "Element Position" to find the structure rather than a "Byte Offset".
					Cortex::Read::fromBinFile(this->characterCacheObjects.at(vectLocForChar)->getFileStreamReferenceForDataRows(), childSigStruct, elementPositionForChildSigStructureInCacheFile * (sizeof Cortex::ChildSignature));

					if(strcmp(childSigStruct.recordName, latestChildSigIndexesVect[x].recordName) != 0){
						throw "The Child Signature Cache File Element Position points at a record structure that doesn't match.";
					}

					byteOffsetForTemp = Cortex::Write::toBinFile(fsTempBin, childSigStruct, byteOffsetForTemp);

					// We just wrote a single Structure to the Full Temp, now we need all of the C27/LX Rows.
					// Other processes could have been writing to the "Cache Files".  This will extract the latest "Row Indexes" for the given CharacterCode/ChildSignature combo into a vector and make sure that they are properly sorted.
					std::vector<Cortex::Index> latestRowIndexesVect;
					this->getLatestRowIndexes(latestRowIndexesVect, loop_CharCode, childSigSearch);

					// Write all of the Row Indexes (belonging to the Child Signature) to the Full-Temp file contiguously.
					// We are going to overwrite them in just a bit.  We need to write them to disk now as a place holder.
					// This variable represents the start of the "place holder".
					std::streamsize bytePositionOfRowIndexesInTemp = byteOffsetForTemp;

					for(unsigned long k=0; k < latestRowIndexesVect.size(); k++){
						byteOffsetForTemp = Cortex::Write::toBinFile(fsTempBin, latestRowIndexesVect[k], byteOffsetForTemp);
					}

					// So there are 28 times more indexes than there are rows.
					// The first section 1/28 is all that we are worried about (they will help us find the Row Structures).
					// The remaining 27/28 indexes just provide other indexes to the same extact data rows (just indexed by different columns names).
					long totalNumberOfDataRows = latestRowIndexesVect.size() / Cortex::Globals::NUMBER_OF_INDEXED_COLUMNS_ON_DATA_ROWS;

					for(long k=0; k < totalNumberOfDataRows; k++){

						// We are fetching the Rows with "random read access" to match the sequence of our indexes (which are ordered alphabetically).
						// They could have become scattered as new records were added during runtime. Because the Data Rows are stored contiguously (without sub-structures), we can use the "Element Position" instead of "Byte Offsets".
						Cortex::Row rowStruct={0};
						Cortex::Read::fromBinFile(this->characterCacheObjects.at(vectLocForChar)->childSignatureCacheObjects.at(vectLocForChildSig)->getFileStreamReferenceForDataRows(), rowStruct, (latestRowIndexesVect[k].elementPosition * (sizeof Cortex::Row)));

						// The "LX" column is our primary index for "Data Rows".  The LX indexes are located within the first stack 1/28. That is why we can access it with the variable "k" here.  .
						if(strcmp(rowStruct.LX, latestRowIndexesVect[k].recordName) != 0){
							throw "The Child Signature Cache File Element Position points at a record structure that doesn't match.";
						}

						byteOffsetForTemp = Cortex::Write::toBinFile(fsTempBin, rowStruct, byteOffsetForTemp);

						// Our various "cache files" may have scattered the "Element Positions" relative to the "Index Positions". We are going to syncronize them in the vector now.
						// The Vector represents "Row Indexes" which were just recently written to the file.  That means that we will have to overwrite the row indexes after looping over the rows inside of here.
						// Because there are many different indexes (C27 + LX) pointing to the same row, we need to coordinate the new Element/Index/ByteOffset through all of them with a "Search & Replace".
						// For example.  L5_G1 and L2 "may" have different index positions in our "Cache Files", but they would still have the same "Element Position" for the "Row Structure" (which is like an auto-increment ID).
						this->searchReplaceRowIndexPosistions(latestRowIndexesVect, latestRowIndexesVect[k].indexPosition, k, latestRowIndexesVect[k].elementPosition, k, latestRowIndexesVect[k].byteOffset);
					}

					// The indexes are organized correctly within our vector now, however we need to go back in the Full-Temp file and re-write the new index information to disk.
					for(unsigned long i=0; i<latestRowIndexesVect.size(); i++){

						// Now change the of the negative element posistions back to their absolute value (required by method searchReplaceRowIndexPosistions).
						if(latestRowIndexesVect[i].elementPosition > 0){
							throw "Except for the element posistion Zero, there should not be any positive Element Positions at this point.";
						}
						latestRowIndexesVect[i].elementPosition *= -1;

						// This "byte counter" variable is reset on every loop (per Child Signature).
						bytePositionOfRowIndexesInTemp = Cortex::Write::toBinFile(fsTempBin, latestRowIndexesVect[i], bytePositionOfRowIndexesInTemp);
					}
				}
				else{

					// ----------- This means that the "Child Signature" is not cached, even though the parent "Character Stucture" is cached. ----------------

					// We can't be sure that the ChildSignature Structures are syncronized with the Index Positions in our master file, or in the "Cortex Cache Files".
					// Use a Method Call do find the ChildSignature in the Master with a Binary Search using "fStream pointers".
					std::streamsize byteOffsetForChildSig = CortexSearch::getByteOffsetFromChildSignatureIndex(this->fileStreamMasterFile, loop_CharCode, this->totalCharactersInMaster, childSigSearch, cortexChar.childSignatureCount);

					// Since we haven't cached it yet, we can just copy the ChildSignature Stucture, Row Indexes, and Rows from the master into the Full Temp.
					// Nothing within that structure (and sub-structures) will change.  All that we are really doing is updating the "Child Signature Indexes" above and moving this entire structure somewhere else in the file.
					Cortex::ByteOffsets byteOffsetAfterChildSigCopy = CortexCopy::ChildSignatureStructureAndRows(this->fileStreamMasterFile, (loop_ByteOffsetForMaster + byteOffsetForChildSig), fsTempBin, byteOffsetForTemp);

					// Advance the file pointer for the FullTemp as a result of the method call above.
					// Don't advance the "master pointer".  That is re-calculated that on every loop.
					byteOffsetForTemp = byteOffsetAfterChildSigCopy.destinationByteOffset;
				}
			}

			// Since this Character Container may have changed as a result of all the contatenation and re-organization, we have to go BACK UP in the Full-Temp file and let the parent container "Character Structure" know what it has inside of it.
			// We don't have to worry about whether ChildSignatures were cached, if they have grown since last time, etc... because however much the Full-Temp File grew from the last loop is how big the Character is.
			cortexChar.byteTalley = byteOffsetForTemp - charCodeStructureLocationInTemp;
			cortexChar.childSignatureCount = latestChildSigIndexesVect.size();

			// Overwrite the Character Code structure in the Full-Temp file with the new info. The size of the structure has increased, so we don't have to worry about shifting any bytes down in the file.
			Cortex::Write::toBinFile(fsTempBin, cortexChar, charCodeStructureLocationInTemp);

			// Since we may have re-organized the "Child Signature Indexes", we must re-write our Vector back to the Full-Temp file.
			std::streamsize byteOffsetForChildSigIndexes = charCodeStructureLocationInTemp + (sizeof Cortex::Character);

			for(unsigned long x=0; x < latestChildSigIndexesVect.size(); x++){
				byteOffsetForChildSigIndexes = Cortex::Write::toBinFile(fsTempBin, latestChildSigIndexesVect[x], byteOffsetForChildSigIndexes);
			}
		}
		else{
			// If the Character is Not cached, extract the entire Character structure (and sub-structures) out of the master and copy into the Temp.
			// The "Byte Talley" and all of the sub indexes should be fine because they store "relative values".
			Cortex::ByteOffsets byteOffsetsAfterCopy = CortexCopy::CharacterStructureAndEverythingWithin(this->fileStreamMasterFile, loop_ByteOffsetForMaster, fsTempBin, byteOffsetForTemp);

			// We are tracking the "Master Byte Offset" separately than the "Destination Byte Offset" now.
			byteOffsetForTemp += byteOffsetsAfterCopy.destinationByteOffset;
		}
	}
}


// Pass in a Vector by reference to have the lastest "C27/LX Row Indexes" filled up.
// It will basically just read the Index Structs out of a single file.  But it will also do some error checking a long the way.
void CortexCache::getLatestRowIndexes(std::vector<Cortex::Index>& rowIndexesVect, std::size_t characterCode, SignatureStruct childSignature)
{
	rowIndexesVect.clear();

	std::size_t vectLocForChar = this->getIndexPositionForInsertionOfCharacterCode(characterCode);

	if(this->characterCacheObjects.at(vectLocForChar)->characterCode != characterCode){
		throw "You can't call getLatestRowIndexes on a Character Code that isn't cached yet.";
	}

	long vectLocForChildSig = this->characterCacheObjects[vectLocForChar]->getIndexPositionForInsertionOfChildSignature(childSignature);

	if(strcmp(this->characterCacheObjects[vectLocForChar]->childSignatureCacheObjects.at(vectLocForChildSig)->recordName, childSignature.digestedStr.c_str()) != 0){
		throw "You can't call getLatestRowIndexes on a Child Signature that isn't cached yet.";
	}

	// Kind of like a "Bubble Sort", we will be comparing the last value to the current as we loop to see if anything is out of sequence.
	char lastSignatureInLoop[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];

	// Make sure there is a Null Character for the last position.
	std::memset(lastSignatureInLoop, 0, sizeof lastSignatureInLoop);

	std::fstream& fsRowsIndex = this->characterCacheObjects[vectLocForChar]->childSignatureCacheObjects[vectLocForChildSig]->getFileStreamReferenceForIndexes();

	// Remember that this will be 28 times greater than the number of elements. It is like a Database Row with 28 columns (each having indexes).
	long numberOfRowIndexesInCacheFile = this->characterCacheObjects[vectLocForChar]->childSignatureCacheObjects[vectLocForChildSig]->getNumberOfElementsInCacheFiles();

	if(numberOfRowIndexesInCacheFile % Cortex::Globals::NUMBER_OF_INDEXED_COLUMNS_ON_DATA_ROWS != 0){
		throw "The total number of Row Indexes must be evenly divisible by the number of columns which are indexed.";
	}

	rowIndexesVect.clear();
	rowIndexesVect.reserve(numberOfRowIndexesInCacheFile);

	std::streamsize tempFileByteOffset = 0;

	for(long i=0; i<numberOfRowIndexesInCacheFile; i++){

		Cortex::Index cortexIndex={0};
		tempFileByteOffset = Cortex::Read::fromBinFile(fsRowsIndex, cortexIndex, tempFileByteOffset);

		rowIndexesVect.push_back(cortexIndex);

		// Don't compare the sorting to the last element unless there we have a "last element" to check.
		if(i > 0){
			if(std::strcmp(cortexIndex.recordName, lastSignatureInLoop) <= 0){
				throw "The Child Signature Indexes do not appear to be sorted correctly within the Cache File in method getLatestRowIndexes().";
			}
		}

		// Copy the C-string so we can make sure the following index item is sorted above this one.
		strncpy(lastSignatureInLoop, cortexIndex.recordName, Cortex::Globals::CORTEX_SIGNATURE_LENGTH);
	}
}


// This will load up the ChildSignature indexes out of the Cache File and it will make sure that there isn't a "Sorting error".
void CortexCache::getLatestChildIndexes(std::vector<Cortex::Index>& childSigIndexesVect, std::size_t characterCode)
{
	childSigIndexesVect.clear();

	std::size_t vectLocForChar = this->getIndexPositionForInsertionOfCharacterCode(characterCode);

	if(this->characterCacheObjects.at(vectLocForChar)->characterCode != characterCode){
		throw "You can't call getLatestChildIndexes on a Character Code which hasn't already been cached.";
	}

	// Kind of like a "Bubble Sort", we will be comparing the last value to the current as we loop to see if anything is out of sequence.
	char lastSignatureInLoop[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];

	// Make sure there is a Null Character for the last position.
	std::memset(lastSignatureInLoop, 0, sizeof lastSignatureInLoop);

	std::fstream& fsChildSigIndex = this->characterCacheObjects[vectLocForChar]->getFileStreamReferenceForIndexes();

	long numberOfIndexesInCacheFile = this->characterCacheObjects[vectLocForChar]->getNumberOfElementsInCacheFiles();


	childSigIndexesVect.clear();
	childSigIndexesVect.reserve(numberOfIndexesInCacheFile);

	std::streamsize tempFileByteOffset = 0;

	for(long i=0; i<numberOfIndexesInCacheFile; i++){

		Cortex::Index cortexIndex={0};
		tempFileByteOffset = Cortex::Read::fromBinFile(fsChildSigIndex, cortexIndex, tempFileByteOffset);

		childSigIndexesVect.push_back(cortexIndex);

		// Don't compare the sorting to the last element unless there we have a "last element" to check.
		if(i > 0){
			if(std::strcmp(cortexIndex.recordName, lastSignatureInLoop) <= 0){
				throw "The Child Signature Indexes do not appear to be sorted correctly within the Cache File in method getLatestChildIndexes().";
			}
		}

		// Copy the C-string so we can make sure the following index item is sorted above this one.
		strncpy(lastSignatureInLoop, cortexIndex.recordName, Cortex::Globals::CORTEX_SIGNATURE_LENGTH);
	}
}

// CAUTION, this will leave all of the "Element Positions" set to a negative value after doing the search and replace.
// This is needed to avoid collisions between the "Replace Element" on a previous loop with the "Search Element" on a future loop.
void CortexCache::searchReplaceRowIndexPosistions(std::vector<Cortex::Index>& rowIndexesVect, long searchIndexPos, long replaceIndexPos, long searchElementPos, long replaceElementPos, std::streamsize searchByteOffset)
{
	if(rowIndexesVect.size() % Cortex::Globals::NUMBER_OF_INDEXED_COLUMNS_ON_DATA_ROWS != 0){
		throw "The total number of Row Indexes must be evenly divisible by the number of columns which are indexed in method searchReplaceRowIndexPosistions.";
	}

	long totalNumberOfDataRows = rowIndexesVect.size() / Cortex::Globals::NUMBER_OF_INDEXED_COLUMNS_ON_DATA_ROWS;

	// Coordinate the Element/Index re-ordering for all of the indexes on the various columns.
	for(long indexCluster = 0; indexCluster < Cortex::Globals::NUMBER_OF_INDEXED_COLUMNS_ON_DATA_ROWS; indexCluster++){

		bool foundRowMatch = false;

		for(long relativeIndexPosition = 0; relativeIndexPosition < totalNumberOfDataRows; relativeIndexPosition++){

			long absoluteIndexPosition = (indexCluster * totalNumberOfDataRows) + relativeIndexPosition;

			if(rowIndexesVect[absoluteIndexPosition].elementPosition == searchElementPos){

				if(foundRowMatch){
					throw "There is a problem with the indexes.  There should only be 1 matching element positions within in the loop.";
				}
				if(rowIndexesVect[absoluteIndexPosition].indexPosition != searchIndexPos){
					throw "The indexes seem to have been mixed up.  The element positions match, but the index positions don't.";
				}
				if(rowIndexesVect[absoluteIndexPosition].byteOffset != searchByteOffset){
					throw "The indexes seem to have been mixed up.  The element * Index positions match, but the Byte Offsets don't.";
				}

				// Since "long" values can hold negative values... change the "Element Position" to a negative temporarily
				// Otherwise we could have a collission during the search and replace.  One of the Replaced Values could match up to a future Search.
				rowIndexesVect[absoluteIndexPosition].elementPosition = replaceElementPos * -1;
				rowIndexesVect[absoluteIndexPosition].indexPosition = replaceIndexPos;

				// Since the "Row Structures" are all the same size (with no sub-structures), the Byte Offset can be inferred from the Element Position (which is like an Auto-Increment ID).
				rowIndexesVect[absoluteIndexPosition].byteOffset = replaceElementPos * (sizeof Cortex::Row);

				foundRowMatch = true;
			}
		}

		if(!foundRowMatch){
			throw "Could not find a row match for an index in method searchReplaceRowIndexPosistions";
		}
	}
}

