#pragma once


// Create a name space to make the compiler's auto-complete work better.
namespace Cortex{

	struct Globals
	{
		const static std::string CORTEX_HEADER_DESC;

		// Make sure there is enough characters to accomodate all unicode chars.
		const static int MAX_CHARACTER_COUNT = 65535;

		// 28 + NullByte
		// We can't use "string" objects because these structures will be stored in the DB.
		const static int CORTEX_SIGNATURE_LENGTH = 28;
		const static int CORTEX_HEADER_DESC_LENGTH = 15;

		// We need to know how many Indexes there are for the "Data Rows" because all of the indexes will be stacked on top of each other.
		// The first cluster 1/28 will be the LX Columns (or the primary Context Signature).
		const static int NUMBER_OF_INDEXED_COLUMNS_ON_DATA_ROWS = 28;

		// The main binary interface.
		const static std::string CORTEX_MASTER_BINARY_FILE;

		// Used by the CortexCache consolidation routine.
		const static std::string CORTEX_TEMPORARY_BINARY_FILE;

		// Converts an INT to a string.
		static std::string getCharCodeString(std::size_t charCode);

		// Pass in a string by reference like "bca" and it will change it to "abc".
		static void sortCharactersInString(std::string& strToSrt);

		// Fstreams don't create files. This will truncate the given filename (if exists) and open the fstream handle with it.
		static void resetFile(std::fstream& fileHandle, std::string fileName, bool isBinary);

	};



	// This structure will only be used 1 time within the beggining of the file.
	struct Header
	{
		// Leave 1 for the NullByte
		char headerDesc[Cortex::Globals::CORTEX_HEADER_DESC_LENGTH + 1];
		std::streamsize byteTalley;
		std::size_t characterCount;
	};

	struct Character
	{
		char character;
		std::size_t characterCode;

		// This will contain a Byte Talley of all "Child Signatures", the "Child Signature Indexes", "Signature Rows", and "Signature Indexes"
		// The Byte Talley will not include its own indexes.  The main "Header" will sit on top of the those (which are above all Character Code Structs).
		// The Byte Talley will include the size of this structure.
		std::streamsize byteTalley;
		long childSignatureCount;
	};

	struct ChildSignature
	{
		std::size_t characterCode;

		char recordName[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];

		// The Byte Talley will not include its own indexes.
		// "ChildSignature Indexes" are found after the "Character Structure" and before the collection of ChildSignature Objects.
		// The Byte Talley will include the size of this structure.
		// It will also include All of the Indexes and Rows which sit underneath (before the next 'Child Signature is reached).
		std::streamsize byteTalley;

		// The 'Row Count' will be used to tell us how many "Signature Rows" there are.
		// There are 27 separate indexes within each "Child Signature".  The 'Row Count' will also indicate the length for all of those indexes as well.
		long rowCount;
	};

	// This is a general container to be used with "Characters", "ChildSignatures", and C27's
	struct Index
	{
		// This could be a "character code", or a digestive hash of a particular signature.
		char recordName[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];

		// This will contain the amount of bytes offset from the parent container where the record structure can be found.
		// The offset starts from the point in which the "Index Collection" ends.
		// So the first record in the list will show a "byteOffset" of Zero.
		std::streamsize byteOffset;

		// If there are 5 indexes within an "Index Collection", there must be values 0-4 stored in here corresponding to the alpha-numeric sorting of the "record name".
		// This is equivalent to an auto-increment ID.  The "indexPosition" will be related to the alpha-numeric positions of the index.  The elements will always be inserted at the bottom.
		long elementPosition;

		// The "array index" corresponds with the position in the "Parent Container".  The indexes are store alpha-numberically, but the actual rows may be scattered.
		long indexPosition;
	};


	// TODO:  Maybe we need to put an "Access Counter" the Rows
	// Spamming the system won't hurt the logic... just the File Size.
	// We can do things like "Prune Character Sets"... but how to prune junk??????
	// The "Child Signatures" would prevent most junk from getting accessed.
	// Maybe there are extra time stamps?  That way each person could know...
	// - Created
	// - Imported to your database
	// - Has Been Accessed by me (Y/N)
	// Maybe that is all we need to know?  After a year of using the system... if you pruned off every row that you never used... you wouldn't care.
	// Therefore, your "My Cortex File" should have a record of whether you touched it or not.
	// Ahh... Then you could say "Only My Knowledge"... or complete database too!
	// Actually, the file system should do it automatically.  No command line switches.  2 Different Text-Based backups.

	struct Row
	{
		// This is redundant (as long as the database is not corrupted).
		std::size_t characterCode;

		// This child signature comes from above.  At the very top it made from digesting the "InventorID" and the "Flip/Flop Bit".
		char childSignature[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];

		// A unix timestamp value.
		unsigned long timeStamp;

		// This will be set to "FALSE" when you import someone elses brain.
		// If you have conversations which touch certain rows, switch it to true.  By "touch", meaning if either the LX or C27 Query matched the record.
		// This is how we can filter out spam.  Although Spam won't bother the user experience, it can increase file size.
		// After a year of using the system you will have "touched" quite a lot of stuff.
		// If you deleted all rows which weren't touched by you, the net result is a smaller file ... and ... gNet will work just as good for you tommorow.
		bool touchedByMe;

		// Any time a record is inserted with an L1 Query, this bit will get set to TRUE.
		// This will help people share "only their knowledge" with others.
		// That might be useful for quite a bit more than just reducing file-size (for sharing).
		bool addedByMe;

		// This comes from digesting up to 6 context items.
		// It is the "MainSignature" which will be passed down as a "Child Signature" to other levels.
		char LX[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];

		// The C27 Columns
		char c27Columns[Tri::pyramidDepth][Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];

		/*
		char L1[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L2_G4[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L2_G3[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L3_G3[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L2_G2[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L3_G21[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L3_G2[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L4_G2[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L2_G1[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L3_G12[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L3_G11[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L4_G11[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L3_G1[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L4_G101[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L4_G1[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L5_G1[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L2[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L3_G03[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L3_G02[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L4_G02[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L3_G01[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L4_G01[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L5_G01[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L3[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L4[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L5[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		char L6[Cortex::Globals::CORTEX_SIGNATURE_LENGTH + 1];
		*/
	};

	// Some methods will copy data between files.
	// They need to return the new "Byte Offsets" from both, which means a single return value will not suffice.
	struct ByteOffsets
	{
		std::streamsize sourceByteOffset;
		std::streamsize destinationByteOffset;
	};


	// If we had all of the structures inherit from this, that would increase the size of the structure because a pointer to the method would be stored.
	// So let's use static methods with values passed by reference.
	class Write
	{
	public:
		// Pass in the "ByteOffset" for writing to disk and it will return the "ByteOffset" after adding the size of its own structure.
		// By overloading, we don't have to worry about which one to call.
		static std::streamsize toBinFile(std::fstream& fileH, const Cortex::Header& cortexHeader, std::streamsize byteOffset);
		static std::streamsize toBinFile(std::fstream& fileH, const Cortex::Character& cortexChar, std::streamsize byteOffset);
		static std::streamsize toBinFile(std::fstream& fileH, const Cortex::ChildSignature& cortexChildSig, std::streamsize byteOffset);
		static std::streamsize toBinFile(std::fstream& fileH, const Cortex::Index& cortexIndex, std::streamsize byteOffset);
		static std::streamsize toBinFile(std::fstream& fileH, const Cortex::Row& cortexRow, std::streamsize byteOffset);
	};

	class Read
	{
	public:
		// Pass in the "ByteOffset" for where the structure should be read from (relative to the begginning of the file) and it will increment the "ByteOffset" and return it.
		// The Structure will be filled up (by reference).
		static std::streamsize fromBinFile(std::fstream& fileH, const Cortex::Header& cortexHeader, std::streamsize byteOffset);
		static std::streamsize fromBinFile(std::fstream& fileH, const Cortex::Character& cortexChar, std::streamsize byteOffset);
		static std::streamsize fromBinFile(std::fstream& fileH, const Cortex::ChildSignature& cortexChildSig, std::streamsize byteOffset);
		static std::streamsize fromBinFile(std::fstream& fileH, const Cortex::Index& cortexIndex, std::streamsize byteOffset);
		static std::streamsize fromBinFile(std::fstream& fileH, const Cortex::Row& cortexRow, std::streamsize byteOffset);
	};
}


