#pragma once


class Context
{
public:

	static const int MAX_STRING_SIZE = 255;
	static const int MAX_CONTEXT_LENGTH = 6;
	static const int START_CONTEXT = 1;

	// This will be used to split up the character stream.
	const static char END_SYMBOL;

	Context();

	// It is very common to have an "Empty Context Signature".
	// We can just define it ahead of time and save the "digestive hash routine" a LOT of work, even though the task is SO small.
	SignatureStruct getEmptySignature();

	// Pass in a string which has memory allocated with "new".
	// Adding a "Context Item" could push another one off of the end (if the array already has 6).
	// If so, it will free the memory for the string (which falls off the end).
	// The closer to the front of the array, the newer the context.
	void addContextItem(std::string* data);

	// If you have a string (but haven't allocated memory on the heap), pass in the handle and this method will make a copy on the heap.
	void addContextItemByCopy(std::string& data);

	// Overload the method.
	void addContextItemByCopy(const char* dataStr);

	const int getNumberOfContextItems();

	// This will free the memory for the strings too.
	void resetContext();

	void clearContextStreamLock();

	bool isEmpty();

	// The position is "1-based".
	// Do not go past the value for "getNumberOfContextItems()"
	const std::string* getContextItemByPosition(const int position);

	// Returns the signature for the Context Items (even if the object is empty).
	const SignatureStruct getSignature_Full();

	// Returns the signature for the last Context Item.
	// Make sure that the object is not empty before calling this method.
	const SignatureStruct getSignature_L1();

	// Application is single-threaded, so using a Static object to calculate the context signatures will increase performance a bit.
	static Signature signatureObj;

	// Returns a number between 1 and 27.
	const int getNumberOfC27Queries();

	// Make sure that the number doesn't go past the getNumberOfC27Queries() value.
	// Value is 1-based.
	const QueryStruct getC27QueryByPriority(const int priorityVal);

	// This will set the character stream from the context object and return a reference to the string.
	// It is safe to clear the context after acquring the pointer to the string.
	const std::string* getCharacterStream();

	// Based upon the items in contex, this will return a "Cortex Row" which can be inserted into the Database.
	// The default unixTimestamp will be now();
	Cortex::Row getCortexRow(std::size_t charCode, SignatureStruct childSignature, time_t unixTimeStamp = 0);

	// Will free memory for string objects.
	~Context();

private:

	bool contextStreamLock;

	int contextItemsCount;

	bool querySet_L1;
	bool querySet_Full;
	bool querySet_C27;

	void setC27Query();
	int numberC27Queries;

	// Holds an array of up to 27 name/value pairs.
	QueryStruct queryObjects_C27[Triangle::MAX_TRIANGLES];

	static bool emptySignatureCalculated;
	static SignatureStruct emptySignatureStruct;

	// These will contain a digestive hash of the respective "Context Items/Values".
	SignatureStruct signatureStruct_L1;
	SignatureStruct signatureStruct_Full;

	// This object will hold the data in memory (no more than 6 items) that correspond with the signatures.
	std::string* contextDataArr[Context::MAX_CONTEXT_LENGTH];

	// The character stream will hold a copy of the data from the "Context Array".
	std::string characterStream;

	// A parrallel array letting us know what positions have been filled.
	bool contextDataFlagsArr[Context::MAX_CONTEXT_LENGTH];

	// Make the Copy Constructor and the Copy Opertator inaccessable.
	Context(Signature const&);
	void operator=(Context const&);
};