#include <iostream>
#include <string>
#include "GapPattern.h"
#include "Triangle.h"
#include "SignatureStruct.h"
#include "Signature.h"
#include "TriangleColumns.h"
#include "QueryStruct.h"
#include "Cortex.h"
#include <ctime>
#include "Context.h"

// Create static object for performance.
Signature Context::signatureObj;

const char Context::END_SYMBOL = '^';


// Hold an "Empty Signature" within static variable.
// We are not going to intialize the "Signature Value" until we call getEmptySignature().
bool Context::emptySignatureCalculated = false;
SignatureStruct Context::emptySignatureStruct;


// Constructor
Context::Context()
{
	// Initialize all of the flags to false (meaning there aren't any context items yet.
	// This is a parrallel array to the one which contains the actual data.
	for(int i=0; i<Context::MAX_CONTEXT_LENGTH; i++){
		this->contextDataFlagsArr[i] = false;
	}

	this->contextStreamLock = false;

	this->contextItemsCount = 0;
	this->numberC27Queries = 0;

	this->querySet_L1 = false;
	this->querySet_Full = false;
	this->querySet_C27 = false;
}


// Allocate the string with "new" and do not delete.
// This class will free the memory when context items "fall off" or the object goes out of scope.
void Context::addContextItem(std::string* data)
{
	if(data->size() > Context::MAX_STRING_SIZE){
		throw "Make sure to trim the data before calling 'addContextItem'";
	}

	// Every time a new "Context Items" is added, the signatures will need to be re-calculated.
	this->querySet_L1 = false;
	this->querySet_Full = false;
	this->querySet_C27 = false;

	// If the context array is already full, we need to remove the last item before shifting everything up.
	// Take the array down a notch to 5
	if(this->contextItemsCount == Context::MAX_CONTEXT_LENGTH){

		delete this->contextDataArr[Context::MAX_CONTEXT_LENGTH -1];

		this->contextDataArr[Context::MAX_CONTEXT_LENGTH -1] = nullptr;

		this->contextDataFlagsArr[Context::MAX_CONTEXT_LENGTH -1] = false;

		// We will be adding a new one shortly.
		this->contextItemsCount--;
	}

	// Shift everything int the array forward before adding our new item to the bottom.
	for(int i = (this->contextItemsCount); i > 0; i--){
		this->contextDataArr[i] = this->contextDataArr[i-1];
		this->contextDataFlagsArr[i] = true;
	}

	// The latest context item will go into the first slot.
	this->contextDataArr[0] = data;
	this->contextDataFlagsArr[0] = true;
	this->contextItemsCount++;
}



// Copies all of the "Context Items" into a character buffer with a single "End Symbol" at the very end.
// It will return a pointer to the character stream member variable for performance.
// The "Context Object" stays alive within the "FamilyTree object" so you don't have to worry about the pointer going out of scope.
// Just make sure not to call this method again until you have used all of the characters.
const std::string* Context::getCharacterStream()
{
	if(this->contextStreamLock)
	{
		throw "You must clear the Lock before calling this method a second time.";
	}

	this->characterStream.clear();

	for(int contextCounter = Context::START_CONTEXT; contextCounter <= this->contextItemsCount; contextCounter++){
		this->characterStream += *this->contextDataArr[contextCounter];
	}

	this->characterStream += Context::END_SYMBOL;

	this->contextStreamLock = true;

	return &this->characterStream;
}


// With all of the recursion going on, it gives some peace of mind to know that the character stream method is not called a second time before we were done working with the last character stream.
void Context::clearContextStreamLock()
{
	this->contextStreamLock = false;
}

// Copy string to the heap and send the pointer into our other method.
void Context::addContextItemByCopy(std::string& data)
{
	std::string *newContextStr = new std::string(data);

	this->addContextItem(newContextStr);
}

// Overload this method so that you can pass in strings manually such as ... contextObj->addContextItemByCopy("Hello");
void Context::addContextItemByCopy(const char* dataStr)
{
	std::string *newContextStr = new std::string(dataStr);

	this->addContextItem(newContextStr);
}


// The full signature has an END symbol between "Context Items" and also one at the very end.
// This method will digest a concatenation of all items.
const SignatureStruct Context::getSignature_Full()
{
	// If case this Conext Object is empty, just return the "Empty Context Signature".
	if(this->contextItemsCount == 0){
		return this->getEmptySignature();
	}

	// In case we call the method repeatadly (without changing the Context Items), return the cached signature.
	if(this->querySet_Full){
		return this->signatureStruct_Full;
	}

	std::string fullContextStr = "";

	for(int i=0; i<this->contextItemsCount; i++){

		fullContextStr += *this->contextDataArr[i];

		fullContextStr += Context::END_SYMBOL;
	}

	// Save the Signature Stuct into our member variable.
	Context::signatureObj.setSignatureByDigesting(fullContextStr.c_str());
	this->signatureStruct_Full = Context::signatureObj.getSignature();

	// Set the Cache Flag.
	this->querySet_Full = true;

	return this->signatureStruct_Full;
}


// This signature does not include the "End Symbol" like the full signature.
const SignatureStruct Context::getSignature_L1()
{
	if(this->contextItemsCount == 0){
		throw "You may not retrieve an L1 signature with an empty Context Object";
	}

	// We might have the signature already calculated and stored within cache.
	if(this->querySet_L1){
		return this->signatureStruct_L1;
	}

	// Create a signature for the last item added to our context obj.
	// All signatures have a trailing "End Symbol" on them.
	// The most recent "Context Item" is always the first element on the data array.
	std::string tempStr = *this->contextDataArr[0] + Context::END_SYMBOL;

	Context::signatureObj.setSignatureByDigesting(tempStr.c_str());

	this->signatureStruct_L1 = Context::signatureObj.getSignature();

	this->querySet_L1 = true;

	return this->signatureStruct_L1;
}


void Context::setC27Query()
{
	if(this->contextItemsCount == 0){
		throw "You may not generate a C27 Query on an empty Context Obj.";
	}

	std::cout << "SETTING the C27 Query -------- << " << std::endl;

	this->numberC27Queries = 0;

	Triangle& triangleObj = Triangle::getInstance();

	std::string tempStr;

	// Go backwards because the lower the number, the lower the priority for matching.
	for(int triangleDepth = Triangle::MAX_TRIANGLES; triangleDepth >= Triangle::START_TRIANGLE; triangleDepth--){

		// Ultimately this decides how many Query items there will be.  It will be somewhere between 1-27.
		// For example, if we only have 2 Context Items... we can't make a "CONTEXT CHAIN" with 3 gaps in it.
		if(triangleObj.getAbsoluteLengthByDepth(triangleDepth) > this->contextItemsCount){
			continue;
		}

		tempStr.clear();

		GapPattern gapStruct = triangleObj.getGapPattern(triangleDepth);

		for(int contextCounter = Context::START_CONTEXT; contextCounter <= this->contextItemsCount; contextCounter++){

			// If there is NOT a "Gap", then include the "Context Item" within a string that we will digest for our signature.
			if(!gapStruct.gaps[contextCounter - 1]){

				tempStr += *this->getContextItemByPosition(contextCounter);
				tempStr += Context::END_SYMBOL;
			}
		}

		//std::cout << "TempStr (" << triangleObj.getColumnNameByDepth(triangleDepth) << ") " << tempStr << std::endl;

		if(tempStr.empty()){
			throw "Did not find any context items";
		}

		// Digest our "Context Stream" (which is like an MD5).
		Context::signatureObj.setSignatureByDigesting(tempStr.c_str());

		QueryStruct queryField;
		queryField.columnEnum = Tri::getEnum(triangleObj.getColumnNameByDepth(triangleDepth));
		queryField.signature = Context::signatureObj.getSignature();

		// Zero-based Array.  The highest priority queries will be stored at the begginning of the array.
		this->queryObjects_C27[this->numberC27Queries] = queryField;

		this->numberC27Queries++;
	}

	this->querySet_C27 = true;
}

// Value is one-based.  Make sure not to exceed the value for getNumberOfQueries().
// Make sure that this C27Query object does not go out of scople before using the QueryStruct!
const QueryStruct Context::getC27QueryByPriority(const int priorityVal)
{
	// This will only calculate the C27 query 1 time (until the "Context Items" change);
	if(!this->querySet_C27){
		this->setC27Query();
	}

	if(priorityVal > this->numberC27Queries){
		throw "The priority value is out of range.";
	}

	// Our internal array is 0-based.
	return this->queryObjects_C27[priorityVal -1];
}


const std::string* Context::getContextItemByPosition(const int position)
{

	if(position > Context::MAX_CONTEXT_LENGTH || position < Context::START_CONTEXT){
		throw "The context item position is out of range";
	}

	if(!this->contextDataFlagsArr[position -1]){
		throw "Make sure to verify that the context position is occupied before calling this method.";
	}

	return this->contextDataArr[position -1];
}

bool Context::isEmpty()
{
	if(this->contextItemsCount == 0){
		return true;
	}
	else{
		return false;
	}
}



void Context::resetContext(){

	// If there are strings on our data array, this will delete the memory for them.
	for(int i=0; i<Context::MAX_CONTEXT_LENGTH; i++){

		// Make sure we have set something first (judging from the parralell array).
		if(this->contextDataFlagsArr[i]){

			// Free memory for "string"
			delete this->contextDataArr[i];

			// Make sure that no junk is left within the array.
			this->contextDataArr[i] = nullptr;

			// Coordinate with the parrallel array.
			this->contextDataFlagsArr[i] = false;
		}
	}

	this->contextItemsCount = 0;
	this->numberC27Queries = 0;

	this->querySet_L1 = false;
	this->querySet_Full = false;
	this->querySet_C27 = false;
}



// Will return the number of "Context Items" set within this object.  By default there is 0.  There can be no more than 6.
const int Context::getNumberOfContextItems(){

	return this->contextItemsCount;
}

const int Context::getNumberOfC27Queries(){

	// If the C27 isn't cached, we will have to re-calculate everything just to find out how many queries there are.
	if(!this->querySet_C27){
		this->setC27Query();
	}

	return this->numberC27Queries;
}

// Digestions of "Empty Signatures" are very common.
SignatureStruct Context::getEmptySignature()
{
	if(emptySignatureCalculated){
		return emptySignatureStruct;
	}

	// Logically an "Empty Signature" should contain a digestion of just the "End Symbol".
	// You can be pretty sure that there won't be a conflict because the "End Symbol" is not allowed for data entry.
	// However, let's set it to something clever instead.  It will still be very unlikely to conflict with valid signature.
	// Just make sure never to calculate "Context Signatures" outside of this class.
	emptySignatureStruct.digestedStr = Signature::CHARACTER_RANGE;

	emptySignatureCalculated = true;

	return emptySignatureStruct;
}

// Based upon the items in contex, this will return a "Cortex Row" which can be inserted into the Database.
// The default unixTimestamp will be now();
Cortex::Row Context::getCortexRow(std::size_t charCode, SignatureStruct childSignature, time_t unixTimeStamp)
{
	if(this->isEmpty()){
		throw "You can not get a 'Cortex Row' out of an empty Context Object.";
	}

	// Default the timestamp to NOW.
	// unixTimeStamp is an optional parameter to this method.
	if(unixTimeStamp == 0){
		unixTimeStamp = time(NULL);
	}

	// Intitialize the POD (Plain Old Data) structure to Null, which will ensure that the Character Arrays contain a NULL bytes.
	// We are going to copy the signatures into the POD without adding a \0 to the end.
	Cortex::Row cortexRow={0};

	// Just copy the child signature into our "Return Row" without inspecting anything.
	std::strncpy(cortexRow.childSignature, childSignature.digestedStr.c_str(), Cortex::Globals::CORTEX_SIGNATURE_LENGTH);

	cortexRow.characterCode = charCode;

	// Copy the timestamp value which will widen the data type to an "unsigned long" integer.
	// We don't have to worry about negative values (before 1970 epoch).
	cortexRow.timeStamp = static_cast<unsigned long>(unixTimeStamp);

	// This is the Main Signature for the Context Object (which will end up matching one of the C27 Field Values).
	std::strncpy(cortexRow.LX, this->getSignature_Full().digestedStr.c_str(), Cortex::Globals::CORTEX_SIGNATURE_LENGTH);

	// CortexRow Stuture was just initialized with {0}, so don't have to worry about assigning "Signature Values" for which this "Context Object" doesn't have a C27 Value for.
	for(int i=Context::START_CONTEXT; i <= this->getNumberOfC27Queries(); i++){

		Tri::Col colEnum = this->getC27QueryByPriority(i).columnEnum;
		std::string sigVal = this->getC27QueryByPriority(i).signature.digestedStr;

		// This will copy the signature into a 2D C-String array.  The first dimension can store up to 27 values, organized by a Triangle Enumeration.
		std::strncpy(cortexRow.c27Columns[colEnum], sigVal.c_str(), Cortex::Globals::CORTEX_SIGNATURE_LENGTH);
	}

	// Assuming that if you are calling this method, the purpose is to insert a new row.
	cortexRow.addedByMe = true;
	cortexRow.touchedByMe = true;

	return cortexRow;
}


// Destructor
Context::~Context()
{
	// If there are strings held within our data array, this will delete the memory for them.
	this->resetContext();
}



