#include <iostream>
#include <string>
#include <fstream>
#include <vector>
#include "SignatureStruct.h"
#include "TriangleColumns.h"
#include "QueryStruct.h"
#include "Cortex.h"
#include "BrainSort.h"


// STATIC Method
// Sorts the characters in a string alpha-numerically by reference.
void BrainSort::sortStr(std::string& strToSrt){

	std::string tempChar1;
	std::string tempChar2;

	bool swapped = false;

	while(true){

		for(std::size_t i=0; i < (strToSrt.size() - 1); i++){

			// Find out if the ASCII code is greater (which are integers ordered alphanumerically
			if(std::size_t(strToSrt[i]) > std::size_t(strToSrt[i+1])){

				swapped = true;

				tempChar1 = strToSrt.substr(i, 1);
				tempChar2 = strToSrt.substr((i+1), 1);

				strToSrt[i+1] = tempChar1[0];
				strToSrt[i] = tempChar2[0];
			}
		}

		if(!swapped){
			break;
		}

		swapped = false;
	}
}

// Returns an index position (starting at Zero) for where the given signature should be inserted.
// If there is already a matching "needle", then this method will return -1.
// The insertion looks for the record which is 1 higher.  If it finds one, it inserts before (which could be at Array[0].
// If it can't find one that is 1 higher, then it will insert at the very end of the array.  In which case, the "Index Position" returned by this method would be equal to the size of the array elements.
long BrainSort::getIndexPositionForInsertion(std::fstream& fileHandle, const std::streamsize startingByteOffset, const SignatureStruct& needle, const long maxRecords)
{
	if(!fileHandle.is_open()){
		throw "The file handle is not open when trying to get index position.";
	}

	// If there are no records yet, then insertion should happen at the first record (zero-based).
	if(maxRecords <= 0){
		return 0;
	}

	fileHandle.seekg(startingByteOffset, std::ios::beg);

	long left = 0;
	long right = maxRecords;


	Cortex::Index cortexIndex={0};

	while(left <= right && left < maxRecords){

		long middle = (left + right) / 2;

		fileHandle.seekp(startingByteOffset + sizeof(Cortex::Index) * middle, std::ios::beg);
		fileHandle.read((char*)&cortexIndex, sizeof(Cortex::Index));

		if(!fileHandle){
			throw "Error in getIndexPositionForInsertion while advancing the read pointer.";
		}

		if(std::strcmp(cortexIndex.recordName, needle.digestedStr.c_str()) == 0){
			// Return Negative 1 because it found a match.
			// We are not supposed to find a match if we are going to insert.
			return -1;
		}
		else if (std::strcmp(cortexIndex.recordName, needle.digestedStr.c_str()) > 0){
			right = middle - 1;
		}
		else{
			left = middle + 1;
		}
	}

	if(left < 0){
		return 0;
	}
	else{
		return left;
	}
}




// Returns the index position matching the record name (which is like an auto-increment) starting at 0.
// If the record can not be found, this method will return -1.
// This is the heart of the "binary search" for all records in the DB.
long BrainSort::getIndexPositionFromRecordName(std::fstream& fileHandle, const std::streamsize startingByteOffset, const SignatureStruct& needle, const long maxRecords)
{
	if(!fileHandle.is_open()){
		throw "The file handle is not open when trying to get index position.";
	}

	// Save a little work.
	if(maxRecords <= 0){
		return -1;
	}

	fileHandle.seekg(startingByteOffset, std::ios::beg);

	long left = 0;
	long right = maxRecords;

	Cortex::Index cortexIndex={0};

	while(left <= right){

		long middle = (left + right) / 2;

		fileHandle.seekp(startingByteOffset + sizeof(Cortex::Index) * middle, std::ios::beg);
		fileHandle.read((char*)&cortexIndex, sizeof(Cortex::Index));

		if(!fileHandle){
			throw "Error in getIndexPositionFromRecordName while advancing the read pointer.";
		}

		if(std::strcmp(cortexIndex.recordName, needle.digestedStr.c_str()) == 0){
			return middle;
		}
		else if (std::strcmp(cortexIndex.recordName, needle.digestedStr.c_str()) > 0){
			right = middle - 1;
		}
		else{
			left = middle + 1;
		}
	}

	return -1;
}



// Will return the "Byte Offset" for a character code.
// The value is relative to where the "Character Indexes" end.
std::streamsize BrainSort::getByteOffsetFromCharacter(std::fstream& fileHandle, char characterNeedle, long totalCharacters)
{

	std::streamsize byteOffset = sizeof(Cortex::Header);

	// The signature for the "Character Collection" is just a single character (versus a digestive hash).
	SignatureStruct characterSignature;
	characterSignature.digestedStr = std::string(1, characterNeedle);

	long indexPosition = BrainSort::getIndexPositionFromRecordName(fileHandle, byteOffset, characterSignature, totalCharacters);

	if(indexPosition < 0){
		throw "The character code could not be located within the index structure.";
	}

	// The Index Structure will contain the "Byte Offset" associated with the Character Stucture.
	Cortex::Index cortexIndex={0};

	// Now it is simple math to extract the Index Stucture.
	fileHandle.seekg(byteOffset + sizeof(Cortex::Index) * indexPosition, std::ios::beg);
	fileHandle.read((char*)&cortexIndex, sizeof(Cortex::Index));

	if(!fileHandle){
		throw "Error reading the file handle in method getByteOffsetFromCharacter";
	}

	return cortexIndex.byteOffset;
}



