#include <iostream>
#include <string>
#include <fstream>
#include <sstream>
#include <vector>
#include "SignatureStruct.h"
#include "TriangleColumns.h"
#include "QueryStruct.h"
#include "Cortex.h"
#include "CortexSearch.h"
#include "CortexCopy.h"
#include "CortexCacheCompartment.h"
#include "CortexCache.h"
#include "BrainIO.h"




BrainIO::BrainIO()
{
	this->cortexCacheObj = &CortexCache::getInstance();
}

bool BrainIO::ifDeffinetlyNotExists(const std::size_t charCode, const SignatureStruct& childSignature, const QueryStruct& theQuery)
{
	//std::fstream fstreamChildSigIndexes = this->cortexCacheObj->getStream_Row_Indexes(charCode, childSignature);

	//CortexSearch::getIndexPositionFromRecordName(fstreamChildSigIndexes, 0,

	return false;
}


Cortex::Row BrainIO::getRow(const std::size_t charCode, const SignatureStruct& childSignature, const QueryStruct& theQuery, const bool touchIt)
{
	long posForIndexFromColumnQuery = this->indexPositionForColumnQuery(charCode, childSignature, theQuery);

	if(posForIndexFromColumnQuery < 0){
		throw "You can not call getRow() on a query that doesn't have a match.";
	}

	std::fstream& fstreamRowIndexes = this->cortexCacheObj->getStream_Row_Indexes(charCode, childSignature);

	// The file size for the "Row Indexes" will let us know how many indexes there are inside.  Remember that there are 28 different columns indexed within the same file.
	std::streamsize totalFileSizeRowIndexes = CortexSearch::getFileSize(fstreamRowIndexes);

	// This will give us the "Byte Offset" within the Index file where our C27/LX cluster begins.
	std::streamsize byteOffsetForColumn = this->getByteOffsetForRowColumnIndex(totalFileSizeRowIndexes, theQuery.columnEnum);

	// This will take us to the proper location within our "Index Cluster".
	byteOffsetForColumn += posForIndexFromColumnQuery * (sizeof Cortex::Index);

	Cortex::Index rowIndex={0};
	Cortex::Read::fromBinFile(fstreamRowIndexes, rowIndex, byteOffsetForColumn);

	if(strcmp(rowIndex.recordName, theQuery.signature.digestedStr.c_str()) != 0){
		throw "The Record name did not match up in method BrainIO::getRow().";
	}

	// Because the Cache Files have isolated the DataRows into their own file, we can use the "Element Position" to quickly find the proper location.
	// Think of the "Element Position" like an auto-increment ID.  The indexes help us find the "ID column" in the data file.
	std::streamsize byteOffsetForRow = rowIndex.elementPosition * (sizeof Cortex::Row);

	std::fstream& fstreamRowData = this->cortexCacheObj->getStream_Row_Data(charCode, childSignature);

	Cortex::Row rowStruct={0};
	Cortex::Read::fromBinFile(fstreamRowData, rowStruct, byteOffsetForRow);

	// Let myself know what I have been thinking about.
	if(touchIt){
		rowStruct.touchedByMe = true;
		Cortex::Write::toBinFile(fstreamRowData, rowStruct, byteOffsetForRow);
	}

	if(strcmp(rowStruct.c27Columns[theQuery.columnEnum], theQuery.signature.digestedStr.c_str()) != 0){
		throw "The Data Row did not match up to the Query in method BrainIO::getRow().";
	}

// TODO:... There could be multiple matches on the C27 Query.  We got to make sure that we have the one with the newest timestamp.

	return rowStruct;
}


bool BrainIO::ifExists(const std::size_t charCode, const SignatureStruct& childSignature, const QueryStruct& theQuery)
{
	if(this->ifDeffinetlyNotExists(charCode, childSignature, theQuery)){
		return false;
	}

	long indexPositionOfIndexStruct = this->indexPositionForColumnQuery(charCode, childSignature, theQuery);

	if(indexPositionOfIndexStruct >= 0){
		return true;
	}
	else{
		return false;
	}
}

// Returns -1 if the ChildSignature does not exist or the Query can not find a match.
// The Index Position will be relative to where the "Index Cluster" begins.  Therefore, the same LX signature may be returned for matching index numbers between L5_01 & L3
long BrainIO::indexPositionForColumnQuery(const std::size_t charCode, const SignatureStruct& childSignature, const QueryStruct& theQuery)
{
	if(!this->childSignatureExists(charCode, childSignature)){
		return -1;
	}

	long countRowIndexes = this->cortexCacheObj->getCurrentRowDataIndexCount(charCode, childSignature);

	std::fstream& fstreamRowIndexes = this->cortexCacheObj->getStream_Row_Indexes(charCode, childSignature);

	// The file size for the "Row Indexes" will let us know how many indexes there are inside.  Remember that there are 28 different columns indexed within the same file.
	std::streamsize totalFileSizeRowIndexes = CortexSearch::getFileSize(fstreamRowIndexes);

	// This will give us the "Byte Offset" within the Index file where our C27/LX cluster begins.
	std::streamsize byteOffsetForColumn = this->getByteOffsetForRowColumnIndex(totalFileSizeRowIndexes, theQuery.columnEnum);

	// The total number of data rows.  For example, if the "Index File" has 28 entries, that means there would be exactly 1 data row.
	long totalDataRows = this->totalDataRows(totalFileSizeRowIndexes);

	return CortexSearch::getIndexPositionFromRecordName(fstreamRowIndexes, byteOffsetForColumn, theQuery.signature, totalDataRows);
}

bool BrainIO::childSignatureExists(const std::size_t charCode, const SignatureStruct& childSignature)
{
	if(childSignature.digestedStr.size() < Cortex::Globals::CORTEX_SIGNATURE_LENGTH){
		throw "The child signature has not been set in method BrainIO::childSignatureExists.";
	}

	long countChildSignatureIndexes = this->cortexCacheObj->getCurrentChildSignatureIndexCount(charCode);
	std::fstream& fileH = this->cortexCacheObj->getStream_ChildSignature_Indexes(charCode);

	// Issolated "Cache Files" are always searched with a Byte Offset of Zero.
	long indexPosOfChildSigIndex = CortexSearch::getIndexPositionFromRecordName(fileH, 0, childSignature, countChildSignatureIndexes);

	if(indexPosOfChildSigIndex >= 0){
		return true;
	}
	else{
		return false;
	}
}


std::streamsize BrainIO::getByteOffsetForRowColumnIndex(const std::streamsize totalFileSizeRowIndexes, const Tri::Col colEnum)
{
	long totalDataRows = this->totalDataRows(totalFileSizeRowIndexes);

	// The Indexes are stored on top of each other contiguously.  All of the L6's are grouped together, next comes L5's, etc.
	// The QueryStruct will have an Enumeration inside (based upon Column Value).
	// Since it is Zero-based, we can find the "chunk of indexes" that you need.
	std::streamsize byteOffset = totalDataRows * colEnum * (sizeof Cortex::Index);

	return byteOffset;
}


long BrainIO::totalDataRows(std::streamsize totalFileSizeRowIndexes)
{
	if(totalFileSizeRowIndexes % (sizeof Cortex::Index) != 0){
		throw "The RowIndex Cache file is corrupted in method BrainIO::totalDataRows() ";
	}

	if(totalFileSizeRowIndexes % ((sizeof Cortex::Index) * Cortex::Globals::NUMBER_OF_INDEXED_COLUMNS_ON_DATA_ROWS) != 0){
		throw "The RowIndex Cache file is corrupted in method BrainIO::totalDataRows().  The number of total indexes is not divisible by the number of Indexed Columns";
	}

	return static_cast<long>(totalFileSizeRowIndexes / ((sizeof Cortex::Index) * Cortex::Globals::NUMBER_OF_INDEXED_COLUMNS_ON_DATA_ROWS));
}


bool BrainIO::insertRow(const Cortex::Row& cortexRow)
{

	std::string L1_sig = cortexRow.c27Columns[Tri::L1];
	std::string LX_sig = cortexRow.LX;

	SignatureStruct childSig;
	childSig.digestedStr = cortexRow.childSignature;

	if(L1_sig.size() < Cortex::Globals::CORTEX_SIGNATURE_LENGTH){
		throw "If you are inserting a row, the L1 column must be populated.";
	}
	if(LX_sig.size() < Cortex::Globals::CORTEX_SIGNATURE_LENGTH){
		throw "If you are inserting a row, the LX Signature must be set.";
	}
	if(childSig.digestedStr.size() < Cortex::Globals::CORTEX_SIGNATURE_LENGTH){
		throw "If you are inserting a row, the LX Signature must be set.";
	}

	if(cortexRow.characterCode == 0){
		throw "The character code must be greater than zero.";
	}

	// This will also make sure that the Character Code is invalid through
	if(!this->childSignatureExists(cortexRow.characterCode, childSig)){
		this->insertChildSignature(cortexRow.characterCode, childSig);
	}

	// If an LX column matches, that guarantees that all of the C27 Columns will match.
	// All that the C27 Columns do is "spread out" the signature over a larger area... and provide indexes to the "LX Signature" from specific "contextual situations" (such as gaps, length, etc.)
	// So what makes the dates important?  Well, you can't have duplicate LX Signatures in the system (we will stop that from happening here).  LX signatures are always unique (relative to the ChildSignature and CharCode).
	// However, you can get multiple "index matches" on an "L2"... so, take whatever one is newest.
// -------  TODO... It seems like the date has to go.... or it has to be relative to the "GNET instalation"
	// Here is the contradiction... what do we do about records that are imported from the Pyramid?
	// --- If we make the timestamps relative to the "user"... then I don't see a problem.  The Pyramid's info would appear "newer" (even if it was entered millions of years ago).
	// The problem is with people who have small cortex files... and add data.
	// The only way for the "relative timestamps" to work is everyone has a "large cortex"
	// Ohh!!! Wait a minute.  I can create a "seed database" that is verified with MD5 and a few records.
	// The "seed database" can record the date in which the records are first imported.
	// Therefore .... ALL Records in the system (for all people) are relative to the date in which they imported them.
	// When we try to import someone else's cortex, we can calculate the "Time Delta" between the date in which they first began using the system.


	std::fstream& fstreamRowIndexes = this->cortexCacheObj->getStream_Row_Indexes(cortexRow.characterCode, childSig);

	// First make sure that the LX record does not exist.

	// The file size for the "Row Indexes" will let us know how many indexes there are inside.  Remember that there are 28 different columns indexed within the same file.
	std::streamsize totalFileSizeRowIndexes = CortexSearch::getFileSize(fstreamRowIndexes);

	// The total number of data rows.  For example, if the "Index File" has 28 entries, that means there would be exactly 1 data row.
	long totalDataRows = this->totalDataRows(totalFileSizeRowIndexes);

	// The LX Record will be found within the first "Index Cluster" (so we start at Byte Offset 0).
	long indexPositionForLxRowInsert = CortexSearch::getIndexPositionForInsertion(fstreamRowIndexes, 0, childSig, totalDataRows);

	// If the method above comes back without a match, then it means that the LX Records already exists.
	// In that case, all of the C27 Columns MUST exist as well.  There is nothing to insert, so return FALSE.
	if(indexPositionForLxRowInsert < 0){
		return false;
	}



	// It seems easier to read everything into a Vector first, but then we have to read across the disk twice (once to read / once to write).
	// We could push the indexes down with a "wave" (using an array buffer), similar to what we did with the ChildSignatures.
	// Maybe I'll come back to that if the application is running sluggish due to TONS of inserts.

	// Create a Vector of Vectors.
	std::vector<std::vector<Cortex::Index>*> vectorIndexClusters;
	vectorIndexClusters.reserve(Cortex::Globals::NUMBER_OF_INDEXED_COLUMNS_ON_DATA_ROWS);

	for(std::vector<std::vector<Cortex::Index>*>::size_type i=0; i < Cortex::Globals::NUMBER_OF_INDEXED_COLUMNS_ON_DATA_ROWS; i++){

		std::streamsize byteOffsetForColumn = this->getByteOffsetForRowColumnIndex(totalFileSizeRowIndexes, Tri::getEnum(i));

		// Because we are storing a Vector of Vectors... we don't want to copy the memory with a push_back().
		std::vector<Cortex::Index>* columnsIndexVect = new std::vector<Cortex::Index>;
		CortexSearch::getIndexesVector(fstreamRowIndexes, *columnsIndexVect, byteOffsetForColumn, totalDataRows);

		vectorIndexClusters.push_back(columnsIndexVect);
	}


	// For each of the index clusters, we are going to find out where to insert each one of the new indexes (in alphabetical order).
	for(std::vector<std::vector<Cortex::Index>*>::size_type i=0; i < Cortex::Globals::NUMBER_OF_INDEXED_COLUMNS_ON_DATA_ROWS; i++){

		Cortex::Index newIndex={0};
		SignatureStruct sigStruct={0};

		if(i==0){
			strncpy(newIndex.recordName, cortexRow.LX, Cortex::Globals::CORTEX_SIGNATURE_LENGTH);
		}
		else{
			// Do (i-1) because the Enum starts on element 0.  The very first index is not part of the C27 Columns.  It is the "Master LX Signature".
			strncpy(newIndex.recordName, cortexRow.c27Columns[Tri::getEnum(i-1)], Cortex::Globals::CORTEX_SIGNATURE_LENGTH);
		}

		sigStruct.digestedStr = newIndex.recordName;

		// We don't need to calculate tye Byte Offsets for TempCache files.  The CortexCache consolidation routine will take care of that.
		newIndex.byteOffset = 0;

		// Because the Element Posistion is 0-based, this will increment our "Total Data Rows" count.
		// Even though the Indexes will be mixed up (between various columns), the Data Rows will be appended to the bottom (like an Auto-increment ID).
		newIndex.elementPosition = totalDataRows;


		long newIndexPosForIndexInVect = CortexSearch::getIndexPositionForVector(*vectorIndexClusters[i], sigStruct);

		// As long as we don't find a matching signature at the position that we are supposed to insert, then we can go ahead and insert it.
		// Otherwise we need to find out which index has a newer date stamp.
//while(long ajustedIndexPosition)

		for(std::vector<Cortex::Index>::size_type x=0; x < vectorIndexClusters[i]->size(); x++){
			vectorIndexClusters[i]->at(x);
		}
	}


	// We are going to write every single Cortex::Index back to the temp file in sequence.  One "index cluster" on top of another.
	std::streamsize indexByteOffset = 0;

	for(std::vector<std::vector<Cortex::Index>*>::size_type i=0; i < Cortex::Globals::NUMBER_OF_INDEXED_COLUMNS_ON_DATA_ROWS; i++){

		for(std::vector<Cortex::Index>::size_type x=0; x < vectorIndexClusters[i]->size(); x++){
			indexByteOffset = Cortex::Write::toBinFile(fstreamRowIndexes, vectorIndexClusters[i]->at(x), indexByteOffset);
		}
	}


	// Clean up the mess we made with the "new" operator on our 2D vector.
	for(std::vector<std::vector<Cortex::Index>*>::size_type i=0; i<Cortex::Globals::NUMBER_OF_INDEXED_COLUMNS_ON_DATA_ROWS; i++){
		delete vectorIndexClusters.at(i);
	}


}



void BrainIO::insertChildSignature(const std::size_t charCode, const SignatureStruct& childSignature)
{
	std::fstream& childSigIndexFileH = this->cortexCacheObj->getStream_ChildSignature_Indexes(charCode);

	long countChildSignatureIndexes = this->cortexCacheObj->getCurrentChildSignatureIndexCount(charCode);

	// If we need to insert an Index, it might be easier to read everything into a vector and then re-write everything.
	// However, that would be a waste if the FileSize is very large and we want to insert something near the bottom.
	long indexInsertForChildSig = CortexSearch::getIndexPositionForInsertion(childSigIndexFileH, 0, childSignature, countChildSignatureIndexes);

	if(indexInsertForChildSig < 0){
		throw "Possible critical race issue in method BrainIO::insertRow.";
	}

	std::streamsize byteOffsetForInsertion = indexInsertForChildSig * (sizeof Cortex::Index);

	// Always initialize POD Structure like this. It will make sure that the C-string has NullBytes everywhere.
	Cortex::Index newIndexStruct={0};

	// We are not going to use byte offsets for the TempCache files.
	// Let the CortexCache consolidation routine figure this out.  We only care about Element/Index positions.
	newIndexStruct.byteOffset = 0;

	// Since the elements are like an auto-increment ID (but Zero based), this will point to the very last record in the Child Signature Data file (advancing the elements by 1).
	newIndexStruct.elementPosition = countChildSignatureIndexes;

	// Self explanatory.
	newIndexStruct.indexPosition = indexInsertForChildSig;

	// The "Child Signature" which we were searching for needs its Hash Signature copied into the new Index Structure.
	strncpy(newIndexStruct.recordName, childSignature.digestedStr.c_str(), Cortex::Globals::CORTEX_SIGNATURE_LENGTH);

	// This will be our Buffer.  We only need one record to shove everything else down like a wave.
	Cortex::Index moveIndexStruct={0};

	// Starting from the "insertion point", we are going to read in a row to our buffer, then then replace it.
	// We will keep forwarding the "Index Structure" all the way to the bottom and increase the filesize by (sizeof Cortex::Index).
	for(long x = indexInsertForChildSig; x <= countChildSignatureIndexes; x++){

		// When "x" is equal to the insertion record, the "New Row" will be written at the ByteOffset.
		// We don't want to write something into its place.
		// As long as we are not on the first itteration within this loop, we can be sure that our "Move Index Struct" will have been filled on a previous loop.
		if(x > indexInsertForChildSig){
			byteOffsetForInsertion = Cortex::Write::toBinFile(childSigIndexFileH, moveIndexStruct, byteOffsetForInsertion);
		}

		// We don't want to "Read" something off of disk which isn't there.
		// Let's say that the current count of "Child Signature" indexes is Zero (before inserting our record).  Our method would have told us to "insert at position 0";
		// In that case, "x" would equal "0" on this one-and-only-loop.  Yet, there is nothing in the file to read.
		if(x < countChildSignatureIndexes){
			Cortex::Read::fromBinFile(childSigIndexFileH, moveIndexStruct, byteOffsetForInsertion);
		}

		// We will insert the new index on the very first loop.
		// It will also advance the file pointer on disk.
		// We can overwrite the first record because we already extracted it into the TempStructure imeediately above.
		if(x == indexInsertForChildSig){
			byteOffsetForInsertion = Cortex::Write::toBinFile(childSigIndexFileH, newIndexStruct, byteOffsetForInsertion);
		}
	}

	// Add the Child Signature Data Structure (which won't have any rows yet).
	Cortex::ChildSignature newChildSigStruct={0};

	// There aren't any Rows yet (since the Child Signature has to come first).  The "Byte Talley" includes the size of itself.
	newChildSigStruct.byteTalley = (sizeof Cortex::ChildSignature);
	newChildSigStruct.characterCode = charCode;
	newChildSigStruct.rowCount = 0;
	strncpy(newChildSigStruct.recordName, childSignature.digestedStr.c_str(), Cortex::Globals::CORTEX_SIGNATURE_LENGTH);

	// This will append the ChildSignature Data Structure at the end.
	std::fstream& childSigDataFileH = this->cortexCacheObj->getStream_ChildSignature_Data(charCode);
	std::streamsize totalChildSigFileSize = CortexSearch::getFileSize(childSigDataFileH);
	totalChildSigFileSize = Cortex::Write::toBinFile(childSigDataFileH, newChildSigStruct, totalChildSigFileSize);

	if(totalChildSigFileSize % (sizeof Cortex::ChildSignature) != 0){
		throw "There appears to be corruption within the ChildSignature Data Cache file.";
	}

	long newChildSignatureDataCount = totalChildSigFileSize / (sizeof Cortex::ChildSignature);
	if(newChildSignatureDataCount != (countChildSignatureIndexes + 1)){
		throw "Possible critical race issue.  The Child Index Count does not match up with the size of the datafile.";
	}
}