#include <iostream>
#include <string>
#include <fstream>
#include <sstream>
#include <vector>
#include "SignatureStruct.h"
#include "TriangleColumns.h"
#include "QueryStruct.h"
#include "Cortex.h"
#include "BrainSort.h"
#include "BrainFetch.h"




Cortex::Header BrainFetch::getHeaderStructure(std::fstream& fileHandle){

	Cortex::Header cortexHeader={0};

	fileHandle.seekg(0, std::ios::beg);
	fileHandle.read((char*)&cortexHeader, sizeof(Cortex::Header));

	if(!fileHandle){
		throw "Error in getHeaderStructure. Can not read the header.";
	}

	if(cortexHeader.headerDesc != Cortex::Globals::CORTEX_HEADER_DESC){
		throw "This file does not appear to be a 'Cortex File'.";
	}

	return cortexHeader;
}

void BrainFetch::getCharacterIndexes(std::fstream& fileHandle, std::vector<Cortex::Index>& indexVector)
{
	long totalChars = BrainFetch::getCharacterCount(fileHandle);
	std::streamsize byteOffset = sizeof(Cortex::Header);

	BrainFetch::getIndexesVector(fileHandle, indexVector, byteOffset, totalChars);
}

void BrainFetch::getIndexesVector(std::fstream& fileHandle, std::vector<Cortex::Index>& indexVector, const std::streamsize startingByteOffset, const long maxRecords)
{
	if(!fileHandle.is_open()){
		throw "The file handle is not open while calling method getIndexesVector.";
	}

	indexVector.clear();
	indexVector.reserve(maxRecords);

	Cortex::Index cortexIndex={0};

	// For each character code, we expect there to be an Index.
	// This will put all of the "Index Structs" into a vector.
	for(long i=0; i<maxRecords; i++){

		fileHandle.seekg(startingByteOffset + sizeof(Cortex::Index) * i, std::ios::beg);
		fileHandle.read((char*)&cortexIndex, sizeof(Cortex::Index));

		if(!fileHandle){
			throw "Error reading the file handle in method getIndexesVector";
		}

		// This will make a copy of the Index Structure... so don't worry about it going out of scope.
		indexVector.push_back(cortexIndex);
	}
}

// Vector is passed by refference to avoid using the "new" operator.
void BrainFetch::getRowsVector(std::fstream& fileHandle, std::vector<Cortex::Row>& rowVector, const std::streamsize startingByteOffset, const long maxRecords){

	if(!fileHandle.is_open()){
		throw "The file handle is not open while calling method getRowsVector.";
	}

	rowVector.clear();
	rowVector.reserve(maxRecords);

	Cortex::Row cortexRow={0};

	for(long i=0; i<maxRecords; i++){

		fileHandle.seekg(startingByteOffset + sizeof(Cortex::Row) * i, std::ios::beg);
		fileHandle.read((char*)&cortexRow, sizeof(Cortex::Row));

		if(!fileHandle){
			throw "Error reading the file handle in method getRowsVector";
		}

		// This will make a copy of the Index Structure... so don't worry about it going out of scope.
		rowVector.push_back(cortexRow);
	}
}

void BrainFetch::getCharacterStuctures(std::fstream& fileHandle, std::vector<Cortex::Character>& characterVector)
{
	if(!fileHandle.is_open()){
		throw "The file handle is not open while calling method getRowsVector.";
	}

	long maxRecords = BrainFetch::getCharacterCount(fileHandle);
	characterVector.clear();
	characterVector.reserve(maxRecords);

	std::streamsize byteOffset = sizeof(Cortex::Header);
	byteOffset += sizeof(Cortex::Index) * maxRecords;

	for(long i=0; i<maxRecords; i++){

		Cortex::Character cortexChar={0};

		fileHandle.seekg(byteOffset, std::ios::beg);
		fileHandle.read((char*)&cortexChar, sizeof(Cortex::Character));

		if(!fileHandle){
			throw "Error reading the file handle in method getCharacterStuctures";
		}

		// This will make a copy of the Index Structure... so don't worry about it going out of scope.
		characterVector.push_back(cortexChar);

		byteOffset += cortexChar.byteTalley;
	}

}


Cortex::Character BrainFetch::getCharacterStruct(std::fstream& fileHandle, char characterNeedle)
{
	long totalCharacterCount = BrainFetch::getCharacterCount(fileHandle);

	std::streamsize characterByteOffset = sizeof(Cortex::Header);
	characterByteOffset += sizeof(Cortex::Index) * totalCharacterCount;

	// Add the 'Byte Offset' where the Destination Character resides on top of our Cortex Header and 'Character Indexes'.
	// The indexes for ChildSignatures will be nested immediately underneath.
	characterByteOffset += BrainSort::getByteOffsetFromCharacter(fileHandle, characterNeedle, totalCharacterCount);

	Cortex::Character cortexChar={0};

	fileHandle.seekg(characterByteOffset, std::ios::beg);
	fileHandle.read((char*)&cortexChar, sizeof(Cortex::Character));

	if(!fileHandle){
		throw "Error reading the file handle in method getCharacterStructure.";
	}

	if(cortexChar.characterCode != std::size_t(characterNeedle)){
		throw "The index appears to be corrupted in method getCharacterStructure";
	}

	return cortexChar;
}

bool BrainFetch::checkIfChildSignatureExists(std::fstream& fileHandle, const char destinationChar, SignatureStruct& childSig)
{

	long totalCharacterCount = BrainFetch::getCharacterCount(fileHandle);

	std::streamsize characterByteOffset = sizeof(Cortex::Header);
	characterByteOffset += sizeof(Cortex::Index) * totalCharacterCount;

	// Add the 'Byte Offset' where the Destination Character resides on top of our Cortex Header and 'Character Indexes'.
	characterByteOffset += BrainSort::getByteOffsetFromCharacter(fileHandle, destinationChar, totalCharacterCount);

	// The indexes for ChildSignatures will be nested immediately underneath the "Character Structure"
	characterByteOffset += sizeof(Cortex::Character);

	Cortex::Character cortexChar = BrainFetch::getCharacterStruct(fileHandle, destinationChar);

	long indexPos = BrainSort::getIndexPositionFromRecordName(fileHandle, characterByteOffset, childSig, cortexChar.childSignatureCount);

	if(indexPos < 0){
		std::cout << "Can not find the Child Signature" << std::endl;
		return false;
	}

	return true;
}

// Returns -1 if the Record Name doesn't exist.
// Make sure that the "Starting Byte Offset" points to the position in the file where the "Index Collection" begins.
long BrainFetch::getElementPositionFromRecordName(std::fstream& fileHandle, const std::streamsize startingByteOffset, const SignatureStruct& recordName, const long maxRecords)
{
	long indexPosition = BrainSort::getIndexPositionFromRecordName(fileHandle, startingByteOffset, recordName, maxRecords);

	if(indexPosition < 0){
		return -1;
	}

	// We can fetch the Index Structure quickly now.
	// The element position will be contained within.
	Cortex::Index cortexIndex = BrainFetch::getIndexStructFromRecordName(fileHandle, startingByteOffset, recordName, maxRecords);

	return cortexIndex.elementPosition;
}

Cortex::Index BrainFetch::getIndexStructFromRecordName(std::fstream& fileHandle, const std::streamsize startingByteOffset, const SignatureStruct& recordName, const long maxRecords)
{

	long indexPosition = BrainSort::getIndexPositionFromRecordName(fileHandle, startingByteOffset, recordName, maxRecords);

	if(indexPosition < 0){
		throw "Can not fetch the Index Structure because the record does not exist.";
	}

	// We can fetch the Index Structure quickly now.
	// The element position will be contained within.
	Cortex::Index cortexIndex={0};

	fileHandle.seekp(startingByteOffset + sizeof(Cortex::Index) * indexPosition, std::ios::beg);
	fileHandle.read((char*)&cortexIndex, sizeof(Cortex::Index));

	if(!fileHandle){
		throw "Error in getIndexStructFromRecordName while advancing the read pointer.";
	}

	return cortexIndex;
}


long BrainFetch::getCharacterCount(std::fstream& fileHandle)
{
	Cortex::Header cortexHeader = BrainFetch::getHeaderStructure(fileHandle);

	return cortexHeader.characterCount;
}


void BrainFetch::dumpFile(std::fstream& fileHandle)
{

	Cortex::Header cortexHeader = BrainFetch::getHeaderStructure(fileHandle);
	std::cout << "------------------------\nStart Cortex Structure\n------------------------\nHeader: " << cortexHeader.headerDesc << "\nByte Talley: " << cortexHeader.byteTalley << "\nCharacter Count: " << cortexHeader.characterCount << "\n------------------------\n";

	std::streamsize byteOffset = sizeof(Cortex::Header);

	// Get all of the "Character Indexes"
	std::vector<Cortex::Index> characterIndexesVect;
	BrainFetch::getIndexesVector(fileHandle, characterIndexesVect, byteOffset, cortexHeader.characterCount);

	std::cout << "\n--- start Character Indexes --- \n";
	for(unsigned long i=0; i<characterIndexesVect.size(); i++){

		Cortex::Index characterIndex = characterIndexesVect.at(i);

		std::string recordDisp = std::string(1, characterIndex.recordName[0]);
		if(std::size_t(characterIndex.recordName[0]) < 33){
			recordDisp = "CTRL";
		}

		std::cout << "CHARACTER INDEX: " << recordDisp << " Index: " << characterIndex.indexPosition << " Element: " << characterIndex.elementPosition  << " Byte Offset: " << characterIndex.byteOffset << "\n";

		byteOffset += sizeof(Cortex::Index);
	}
	std::cout << "---  end  Character Indexes ---\n\n";





	// Get all of the "Character Structures".  Each one of them will have Sub-structures.
	std::vector<Cortex::Character> characterVect;
	BrainFetch::getCharacterStuctures(fileHandle, characterVect);

	for(unsigned long i=0; i<characterIndexesVect.size(); i++){

		// The Character Indexes are parrallel to the Character Structs.
		Cortex::Character characterStruct = characterVect.at(i);

		// Upon each loop (through a Character Code), we want the address directly underneath the Character Structure.
		// That will require adding the Byte Offset of the Character Structure (which can be found in the Character Index).
		std::streamsize bytesOffsetForCharIndex = byteOffset;
		bytesOffsetForCharIndex += sizeof(Cortex::Character);
		bytesOffsetForCharIndex += characterIndexesVect.at(i).byteOffset;

		std::string characterDisplay = std::string(1, characterStruct.character);
		if(std::size_t(characterStruct.character) < 33){
			characterDisplay = "CTRL";
		}

		std::cout << "CHARACTER: " << characterDisplay << " Character Code: " << characterStruct.characterCode << " Byte Talley: " << characterStruct.byteTalley << " Child Signatures: " << characterStruct.childSignatureCount << std::endl;

		// Prevent empty Begin/End delimeters.
		if(characterStruct.childSignatureCount > 0){

			std::cout << "\n     --- Start Child Signature Indexes --- \n";

			std::vector<Cortex::Index> childIndexesVect;
			BrainFetch::getIndexesVector(fileHandle, childIndexesVect, bytesOffsetForCharIndex, characterStruct.childSignatureCount);

			for(unsigned long x=0; x<childIndexesVect.size(); x++){

				Cortex::Index childSignatureIndex = childIndexesVect.at(x);

				std::cout << "     CHILD SIGNATURE INDEX: " << childSignatureIndex.recordName << " Index: " << childSignatureIndex.indexPosition << " Element: " << childSignatureIndex.elementPosition << " Byte Offset: " << childSignatureIndex.byteOffset << "\n";
			}

			std::cout << "     ---  End  Child Signature Indexes --- \n\n";


			std::streamsize bytesOffsetForCharStruct = bytesOffsetForCharIndex + sizeof(Cortex::Index) * characterStruct.childSignatureCount;

			Cortex::ChildSignature childSig={0};


			for(long x=0; x < characterStruct.childSignatureCount; x++){

				bytesOffsetForCharStruct = Cortex::Read::fromBinFile(fileHandle, childSig, bytesOffsetForCharStruct);

				std::cout << "     CHILD SIGNATURE: " << childSig.recordName << " Row Count: " << childSig.rowCount << " Byte Talley: " << childSig.byteTalley << "\n";

			}
			std::cout << std::endl;
		}
	}

	std::cout << "\n------------------------\nEnd Cortex Structure\n------------------------\n\n";


}